/*

This is my C implementation of Maurer's method for Random Bit
Analysis. Anyone using it seriously is *strongly* urged to read the
original article. I have composited several constants, and the significance
of others should be clearly understood before the code is used. Indeed
Maurer describes a whole family of tests: this implements just one.

In the example program, the test is being applied to the least significant 
bit generated by the C rand() function. This will almost certainly fail
the test (my UNIX implementation goes 0101010101010101010.....). A Blum-
Blum-Shub generator should of course pass with flying colours (useful
for calibration)

The test as implemented consumes a lot of output, and so takes a while to
run. As with all statistical tests, the more data, the more confident the 
prognosis. However this data need not be stored.  

The absolute GOOD/BAD indicator should not be taken too seriously. The usual
caveats associated with all statistical testing apply.

Mike Scott

*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Maurer's Universal test for Random bits     *
 * Journal of Cryptology, Vol. 5 1992 pp89-105 */

#define Q 10000       /* > 3000   */
#define K 1000001L    /* > 100*Q  */
#define MEAN 7.1836656
#define DEVIATION 1.5*sqrt(3.238/(double)K)

int maurer(gen)
int (*gen)();
{ /* Universal Test for Randomness */
    double sum,ftu;
    int i; long n;
    static long tab[256];
    for (i=0;i<256;i++) tab[i]=(-1);
    for (n=0;n<Q;n++)   tab[(*gen)()]=n;
    /* check each byte occurred at least once */
    for (i=0;i<256;i++) if (tab[i]<0) return 0;
    sum=0.0;
    for (n=Q;n<Q+K;n++)
    { /* scan byte sequence */
        i=(*gen)();
        sum+=log((double)n-tab[i]);
        tab[i]=n;
    }
    ftu= ((sum/(double)K)/log(2.0));
/*    printf("ft= %lf DEV= %lf\n",ftu-MEAN,DEVIATION); */
    if (ftu>(MEAN+DEVIATION) || ftu<(MEAN-DEVIATION)) return 0;
    return 1;
}

//-----------------------------------------------------------------------------
// Procka w wersji C

    long seed=0;		//koniecznie POZA procka

int generator()
{
    int i,x;
    long temp_x;
    for (x=0,i=0;i<8;i++)
	{
	seed+= GetTickCount();
	seed = (seed * 0x343FD) + 0x269EC3;
	temp_x = (seed >> 0x10 )& 0x7FFF;
	temp_x = (temp_x*temp_x*temp_x) % 0xD76D7249 ;
	x = x<<1;
	x |= ((temp_x & 1));
	}
    return x;
}
//-----------------------------------------------------------------------------

main()
{ /* test bit generator for randomness */
    if (maurer(generator)) 
         printf("This seems to be a GOOD random bit generator\n");
    else printf("This is a BAD random bit generator\n");
} 