package org.t2framework.util.impl;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.t2framework.util.PathUtil;
import org.t2framework.util.UrlTemplate;

import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.PatternUtil;
import commons.util.StringUtil;

/**
 * UrlTemplateImpl is a concrete class of UrlTemplate.
 * 
 * @author shot
 */
public class UrlTemplateImpl implements UrlTemplate {

	protected final String templatePath;

	protected List<Pattern> patterns = CollectionsUtil.newArrayList();

	protected List<String> expressionNames = CollectionsUtil.newArrayList();

	public UrlTemplateImpl(final String templatePath) {
		this.templatePath = templatePath;
		init(templatePath);
	}

	protected void init(final String templatePath) {
		boolean startSlash = PathUtil.startsWithSlash(templatePath);
		String s = escapePatternString(templatePath);
		Matcher matcher = createMatcher(s);
		while (matcher.find()) {
			final String name = matcher.group(2);
			final String expression = matcher.group(1);
			final String[] names = name.split(",", 2);
			if (names.length == 1) {
				s = StringUtil.replace(s, expression, ACCEPTABLE_STR);
				addExpressionName(name);
			} else {
				for (String str : names) {
					addExpressionName(str);
				}
				s = StringUtil.replace(s, expression, ACCEPTABLE_STR);
			}
		}
		patterns.add(PatternUtil.getPattern(s));
		if (startSlash) {
			patterns.add(PatternUtil.getPattern(s.substring(1)));
		} else {
			patterns.add(PatternUtil.getPattern("/" + s));
		}
	}

	protected void addExpressionName(final String name) {
		if (!expressionNames.contains(name)) {
			expressionNames.add(name);
		}
	}

	public List<Pattern> getUrlPattern() {
		return patterns;
	}

	public Map<String, String> parseUrl(String urlPath) {
		Assertion.notNull(urlPath);
		if (!match(urlPath)) {
			return null;
		}
		String path = urlPath;
		if (urlPath.indexOf("/") != 0) {
			path = "/" + urlPath;
		}
		String[] templateStrs = StringUtil.split(templatePath, "/");
		String[] pathStrs = StringUtil.split(path, "/");
		if (templateStrs.length != pathStrs.length) {
			throw new IllegalStateException("should be same depth.");
		}
		Map<String, String> retMap = CollectionsUtil.newHashMap();
		for (int i = 0; i < templateStrs.length; i++) {
			String key = templateStrs[i];
			final int startIndex = key.indexOf("{");
			final int lastIndex = key.lastIndexOf("}");
			if (startIndex < 0 || lastIndex <= 0) {
				continue;
			}
			String value = pathStrs[i];
			if (startIndex == 0 && lastIndex == key.length() - 1) {
				retMap.put(key, value);
			} else {
				final String choppedKey = key.substring(startIndex,
						lastIndex + 1);
				if (startIndex != 0) {
					value = value.substring(startIndex);
				}
				if (lastIndex < key.length() - 1) {
					final String postfix = key.substring(lastIndex + 1);
					int postfixPos = value.lastIndexOf(postfix);
					value = postfixPos > 0 ? value.substring(0, value.length()
							- postfixPos + 1) : value;
				}
				retMap.put(choppedKey, value);
			}
		}
		return retMap;
	}

	@Override
	public boolean match(String urlPath) {
		Assertion.notNull(urlPath);
		for (Pattern p : patterns) {
			if (p.matcher(urlPath).matches()) {
				return true;
			}
		}
		return false;
	}

	@Override
	public String bindByName(Map<String, String> parameters) {
		Assertion.notNull(parameters);
		String s = this.templatePath;
		Matcher matcher = createMatcher(s);
		while (matcher.find()) {
			final String name = matcher.group(2);
			final String[] names = name.split(",", 2);
			if (names.length == 1) {
				s = StringUtil.replace(s, matcher.group(1), parameters
						.get(name));
			} else {
				StringBuilder builder = new StringBuilder();
				for (int i = 0; i < names.length; ++i) {
					final String value = parameters.get(names[i]);
					builder.append(value);
					builder.append(",");
				}
				builder.setLength(builder.length() - 1);
				s = StringUtil.replace(s, matcher.group(1), builder.toString());
			}
		}
		return s;
	}

	@Override
	public String bindByPosition(String... values) {
		String s = this.templatePath;
		Matcher matcher = createMatcher(s);
		if (matcher.groupCount() - 1 > values.length) {
			throw new IllegalStateException();
		}
		int index = 0;
		while (matcher.find()) {
			final String name = matcher.group(2);
			final String[] names = name.split(",", 2);
			if (names.length == 1) {
				s = StringUtil.replace(s, matcher.group(1), values[index]);
			} else {
				StringBuilder builder = new StringBuilder();
				for (int i = 0; i < names.length; ++i) {
					builder.append(values[i]);
					builder.append(",");
				}
				builder.setLength(builder.length() - 1);
				s = StringUtil.replace(s, matcher.group(1), builder.toString());
			}
			index++;
		}
		return s;
	}

	@Override
	public List<String> getExpressionNames() {
		return expressionNames;
	}

	protected Matcher createMatcher(String s) {
		return TEMPLATE_PATTERN.matcher(s);
	}

	protected String escapePatternString(String s) {
		if (s == null) {
			return null;
		}
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < s.length(); i++) {
			char ch = s.charAt(i);
			if (ESCAPED_CHARS.indexOf(ch) >= 0) {
				sb.append('\\');
			}
			sb.append(ch);
		}
		return sb.toString();
	}

	@Override
	public String getTemplatePath() {
		return templatePath;
	}

}
