package org.t2framework.plugin.impl;

import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.t2framework.action.ActionContext;
import org.t2framework.adapter.ContainerAdapter;
import org.t2framework.contexts.WebContext;
import org.t2framework.contexts.WebApplication;
import org.t2framework.navigation.NoOperation;
import org.t2framework.plugin.Plugin;
import org.t2framework.plugin.PluginProcessor;
import org.t2framework.spi.Navigation;

import commons.meta.MethodDesc;
import commons.util.Assertion;

public class PluginProcessorImpl implements PluginProcessor {

	protected final ContainerAdapter<?> containerAdapter;

	public PluginProcessorImpl(ContainerAdapter<?> containerAdapter) {
		this.containerAdapter = Assertion.notNull(containerAdapter);
	}

	@Override
	public void invokeInit(final ServletContext servletContext,
			final WebApplication webApplication) {
		invoke0(servletContext, new Invoker<ServletContext>() {

			@Override
			public ServletContext invoke(Plugin plugin,
					ServletContext servletContext) {
				plugin.initialize(servletContext, webApplication);
				return servletContext;
			}

		});
	}

	@Override
	public void beginRequest(final HttpServletRequest request,
			final HttpServletResponse response) {
		invoke0(request, new Invoker<HttpServletRequest>() {

			@Override
			public HttpServletRequest invoke(Plugin plugin,
					HttpServletRequest request) {
				plugin.beginRequestProcessing(request, response);
				return request;
			}

		});
	}

	@Override
	public void endRequest(final HttpServletRequest request,
			final HttpServletResponse response) {
		invoke0(request, new Invoker<HttpServletRequest>() {

			@Override
			public HttpServletRequest invoke(Plugin plugin,
					HttpServletRequest request) {
				plugin.endRequestProcessing(request, response);
				return request;
			}

		});
	}

	@Override
	public HttpServletRequest createRequest(final HttpServletRequest request,
			final HttpServletResponse response) {
		HttpServletRequest req = invoke0(request,
				new Invoker<HttpServletRequest>() {

					@Override
					public HttpServletRequest invoke(Plugin plugin,
							HttpServletRequest req) {
						return plugin.createRequest(req, response);
					}

				});
		return req;
	}

	@Override
	public HttpServletResponse createResponse(final HttpServletRequest request,
			final HttpServletResponse response) {
		HttpServletResponse res = invoke0(response,
				new Invoker<HttpServletResponse>() {

					@Override
					public HttpServletResponse invoke(Plugin plugin,
							HttpServletResponse response) {
						return plugin.createResponse(request, response);
					}

				});
		return res;
	}

	@Override
	public Navigation invokeComponentCreated(final WebContext context,
			final Object page) {
		return invokeNavigation(page, new NavigationInvoker<Object>() {

			@Override
			public Navigation invoke(Plugin plugin, Object param) {
				return plugin.componentCreated(context, param);
			}

		});
	}

	@Override
	public Navigation beforeActionInvoke(final ActionContext actionContext,
			final MethodDesc targetMethod, final Object page,
			final Object[] args) {
		return invokeNavigation(page, new NavigationInvoker<Object>() {

			@Override
			public Navigation invoke(Plugin plugin, Object param) {
				return plugin.beforeActionInvoke(actionContext, targetMethod,
						param, args);
			}

		});
	}

	@Override
	public Navigation afterActionInvoke(final ActionContext actionContext,
			final MethodDesc targetMethod, final Object page,
			final Object[] args) {
		return invokeNavigation(page, new NavigationInvoker<Object>() {

			@Override
			public Navigation invoke(Plugin plugin, Object param) {
				return plugin.afterActionInvoke(actionContext, targetMethod,
						param, args);
			}

		});
	}

	@Override
	public void beforeNavigation(final WebContext context) {
		invoke0(context, new Invoker<WebContext>() {

			@Override
			public WebContext invoke(Plugin plugin, WebContext ctx) {
				plugin.beforeNavigation(ctx);
				return ctx;
			}

		});
	}

	@Override
	public void afterNavigation(WebContext context) {
		invoke0(context, new Invoker<WebContext>() {

			@Override
			public WebContext invoke(Plugin plugin, WebContext ctx) {
				plugin.afterNavigation(ctx);
				return ctx;
			}

		});
	}

	@Override
	public void destroy(final ServletContext servletContext,
			final WebApplication webApplication) {
		invoke0(servletContext, new Invoker<ServletContext>() {

			@Override
			public ServletContext invoke(Plugin plugin,
					ServletContext servletContext) {
				plugin.destroy(servletContext, webApplication);
				return servletContext;
			}

		});
	}

	protected <T> T invoke0(T org, Invoker<T> invoker) {
		if (!containerAdapter.hasComponent(Plugin.class)) {
			return org;
		}
		List<Plugin> plugins = getPlugins();
		if (plugins == null || plugins.isEmpty()) {
			return org;
		}
		T instance = org;
		synchronized (instance) {
			for (Plugin p : plugins) {
				instance = invoker.invoke(p, instance);
			}
		}
		return instance;
	}

	protected <T> Navigation invokeNavigation(T org,
			NavigationInvoker<T> invoker) {
		if (!containerAdapter.hasComponent(Plugin.class)) {
			return NoOperation.INSTANCE;
		}
		List<Plugin> plugins = getPlugins();
		if (plugins == null || plugins.isEmpty()) {
			return NoOperation.INSTANCE;
		}
		T param = org;
		Navigation navigation = NoOperation.INSTANCE;
		synchronized (param) {
			for (Plugin p : plugins) {
				navigation = invoker.invoke(p, param);
				if (navigation != null || navigation != NoOperation.INSTANCE) {
					break;
				}
			}
		}
		return navigation;
	}

	private static interface Invoker<T> {
		T invoke(Plugin plugin, T param);
	}

	private static interface NavigationInvoker<T> {
		Navigation invoke(Plugin plugin, T param);
	}

	@Override
	public List<Plugin> getPlugins() {
		return containerAdapter.getComponents(Plugin.class);
	}

}
