package org.t2framework.plugin;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.t2framework.action.ActionContext;
import org.t2framework.contexts.WebContext;
import org.t2framework.contexts.WebApplication;
import org.t2framework.spi.Navigation;

import commons.meta.MethodDesc;

/**
 * Plugin is an extension point to extend and update T2 behavior for some
 * reason.Plugin has phases below to extend:
 * 
 * <pre>
 *  - initialize              : Initialize phase as Filter.init().
 *  - beginRequestProcessing  : Begin request processing as the first point of Filter.doFilter().
 *  - createRequest           : Create request.
 *  - createResponse          : Create response.
 *  - componentCreated        : Create the Page instance.
 *  - beforeActionInvoke      : Just before invoking the Page's action method.
 *  - afterActionInvoke       : Just after invoking the Page's action method.
 *  - beforeNavigation        : Just before executing Navigation like forward or redirect.
 *  - afterNavigation         : Just after executing Navigation.
 *  - endRequestProcessing    : End request processing as the end point of Filter.doFilter().
 *  - destroy                 : Destroy phase as Filter.destroy().
 * </pre>
 * 
 * The user should extend {@code org.t2framework.plugin.AbstractPlugin} instead
 * of {@code org.t2framework.plugin.Plugin} because of compatibility matters.
 * 
 * @author shot
 * @see org.t2framework.plugin.AbstractPlugin
 */
public interface Plugin {

	/**
	 * Initialize phase. <br /> Initialize method is expected to configure whole
	 * web application itself.
	 * 
	 * @param servletContext
	 *            : ServletContext
	 * @param webApplication
	 *            : WebApplication as the holder of all Page meta information.
	 */
	void initialize(final ServletContext servletContext,
			final WebApplication webApplication);

	/**
	 * Begin request processing.At that phase, there is no WebContext created,
	 * simply HttpServletRequest and HttpServletResponse are provided.
	 * 
	 * @param request
	 *            : HttpServletRequest
	 * @param response
	 *            : HttpServletResponse
	 */
	void beginRequestProcessing(final HttpServletRequest request,
			final HttpServletResponse response);

	/**
	 * Create HttpServletRequest to use.If needs to wrap HttpServletRequest,
	 * this is the best phase to do.
	 * 
	 * @param request
	 *            : HttpServletRequest
	 * @param response
	 *            : HttpServletResponse
	 * @return wrapped HttpServletRequest.
	 */
	HttpServletRequest createRequest(final HttpServletRequest request,
			final HttpServletResponse response);

	/**
	 * Create HttpServletResponse to use.If needs to wrap HttpServletResponse,
	 * this is the best phase to do.
	 * 
	 * @param request
	 *            : HttpServletRequest
	 * @param response
	 *            : HttpServletResponse
	 * @return wrapped HttpServletResponse.
	 */
	HttpServletResponse createResponse(final HttpServletRequest request,
			final HttpServletResponse response);

	/**
	 * The phase of creating page instance. If extended this method returns
	 * Navigation, except NoOperation, then T2 execute Navigation.execute()
	 * immediately.If so, the action method is not invoked at all.
	 * 
	 * @param context
	 *            : WebContext
	 * @param page
	 *            : created the page instance
	 * @return Navigation instance.
	 */
	Navigation componentCreated(final WebContext context, final Object page);

	/**
	 * Just before action invoking. For instance to use this phase, you can
	 * check the page instance if it passes external requirements that the web
	 * application might have like authentication check. If you do not want T2
	 * invoke the target method, return Navigation instance except NoOperation.
	 * 
	 * @param actionContext
	 *            : ActionContext, the context object that lives in the action
	 *            invoking phase.
	 * @param targetMethod
	 *            : MethodDesc, the target method meta instance.
	 * @param page
	 *            : the page instance
	 * @param args
	 *            : arguments for the target method.
	 * @return Navigation instance.
	 */
	Navigation beforeActionInvoke(final ActionContext actionContext,
			final MethodDesc targetMethod, final Object page,
			final Object[] args);

	/**
	 * Just after action invoking. You can do like overriding Navigation
	 * instance that is invoked by the target method, or cleaning up some
	 * properties that the page instance might have.
	 * 
	 * @param actionContext
	 *            : ActionContext, the context object that lives in the action
	 *            invoking phase.
	 * @param targetMethod
	 *            : MethodDesc, the target method meta instance.
	 * @param page
	 *            : the page instance
	 * @param args
	 *            : arguments for the target method.
	 * @return Navigation instance.
	 */
	Navigation afterActionInvoke(final ActionContext actionContext,
			final MethodDesc targetMethod, final Object page,
			final Object[] args);

	/**
	 * Before Navigation instance, which is given by the result of action
	 * invoked, executing.
	 * 
	 * @param context
	 *            : WebContext
	 */
	void beforeNavigation(final WebContext context);

	/**
	 * After Navigation instance executing.
	 * 
	 * @param context
	 *            : WebContext
	 */
	void afterNavigation(final WebContext context);

	/**
	 * The phase of end request processing. WebContext is also alive, so you can
	 * use WebContext.get() to get whole information.
	 * 
	 * @param request
	 *            : HttpServletRequest
	 * @param response
	 *            : HttpServletResponse
	 */
	void endRequestProcessing(final HttpServletRequest request,
			final HttpServletResponse response);

	/**
	 * The Destroy phase like Filter.destroy().
	 * 
	 * @param servletContext
	 *            : ServletContext
	 * @param webApplication
	 *            : WebApplication as the holder of all Page meta information.
	 */
	void destroy(final ServletContext servletContext,
			final WebApplication webApplication);

}
