package org.t2framework.filter;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.t2framework.action.ActionInvoker;
import org.t2framework.action.impl.ActionInvokerImpl;
import org.t2framework.adapter.ContainerAdapter;
import org.t2framework.contexts.PageDesc;
import org.t2framework.contexts.WebContext;
import org.t2framework.contexts.WebApplication;
import org.t2framework.filter.Initializer.InitContext;
import org.t2framework.handler.GlobalExceptionHandler;
import org.t2framework.navigation.PassThrough;
import org.t2framework.plugin.PluginProcessor;
import org.t2framework.plugin.impl.PluginProcessorImpl;
import org.t2framework.spi.AnnotationResolverCreator;
import org.t2framework.spi.Navigation;

import commons.Disposable;
import commons.Disposer;
import commons.util.Assertion;
import commons.util.Logger;

/**
 * The entry point for T2.
 * 
 * @author shot
 */
public class T2Filter implements Filter {

	public static final String DEFAULT_ENCODING = "UTF-8";

	protected Logger log = Logger.getLogger(T2Filter.class);

	protected Lock lock = new ReentrantLock();

	protected WebApplication webapp;

	protected String[] excludes;

	protected ContainerAdapter<?> adapter;

	protected GlobalExceptionHandler globalExceptionHandler;

	protected AnnotationResolverCreator resolverCreator;

	protected PluginProcessor processor;

	protected FilterConfig config;

	public T2Filter() {
		Disposer.add(new Disposable() {

			@Override
			public void dispose() {
				webapp = null;
				excludes = null;
				adapter = null;
				globalExceptionHandler = null;
				resolverCreator = null;
				processor = null;
				config = null;
			}

		});
	}

	public void init(final FilterConfig config) throws ServletException {
		Assertion.notNull(config);
		final Initializer initializer = new Initializer();
		lock.lock();
		try {
			this.config = config;
			InitContext initContext = initializer.execute(config);
			this.resolverCreator = initContext.resolverCreator;
			this.adapter = initContext.adapter;
			this.webapp = initContext.webapplication;
			this.globalExceptionHandler = initContext.globalExceptionHandler;
			this.excludes = initContext.excludes;
			this.processor = new PluginProcessorImpl(initContext.adapter);
		} catch (Throwable t) {
			if (globalExceptionHandler != null) {
				globalExceptionHandler.handleException(t, null);
			} else {
				throw new ServletException(t);
			}
		} finally {
			try {
				if (processor != null) {
					processor.invokeInit(config.getServletContext(), webapp);
				}
			} finally {
				lock.unlock();
			}
		}
	}

	public void doFilter(final ServletRequest request,
			final ServletResponse response, final FilterChain chain)
			throws IOException, ServletException {
		HttpServletRequest req = (HttpServletRequest) request;
		HttpServletResponse res = (HttpServletResponse) response;
		setupEncoding(req);
		processor.beginRequest(req, res);
		WebContext context = null;
		try {
			req = processor.createRequest(req, res);
			res = processor.createResponse(req, res);
			context = webapp.createContext(req, res, chain);
			if (isExcludeRequest(req)) {
				PassThrough.pass().execute(context);
				return;
			}
			final Map<String, PageDesc> pageDescMap = webapp.getPageDescMap();
			ActionInvoker invoker = new ActionInvokerImpl(context, pageDescMap,
					adapter, resolverCreator, processor);
			if (!invoker.match()) {
				handleOtherRequest(context);
				return;
			}
			final Navigation navigation = invoker.invoke();
			try {
				processor.beforeNavigation(context);
				navigation.execute(context);
			} finally {
				processor.afterNavigation(context);
			}
		} catch (Throwable t) {
			globalExceptionHandler.handleException(t, context);
		} finally {
			processor.endRequest(req, res);
			WebContext.clear();
		}
	}

	public void destroy() {
		processor.destroy(config.getServletContext(), webapp);
		adapter.destroy();
	}

	protected boolean isExcludeRequest(HttpServletRequest req) {
		String uri = req.getRequestURI();
		for (String exclude : excludes) {
			if (uri.endsWith(exclude)) {
				return true;
			}
		}
		return false;
	}

	protected void handleOtherRequest(final WebContext context)
			throws Exception {
		PassThrough.pass().execute(context);
	}

	protected void setupEncoding(HttpServletRequest request)
			throws UnsupportedEncodingException {
		if (request.getCharacterEncoding() == null) {
			request.setCharacterEncoding(webapp.getEncoding());
		}
	}

}
