package org.t2framework.contexts.impl;

import java.io.File;
import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.t2framework.adapter.ContainerAdapter;
import org.t2framework.annotation.core.Page;
import org.t2framework.contexts.Application;
import org.t2framework.contexts.Chain;
import org.t2framework.contexts.PageDesc;
import org.t2framework.contexts.Request;
import org.t2framework.contexts.Response;
import org.t2framework.contexts.Session;
import org.t2framework.contexts.WebContext;
import org.t2framework.contexts.WebApplication;
import org.t2framework.exception.RootPackageNotFoundException;
import org.t2framework.spi.AnnotationResolverCreator;
import org.t2framework.util.UrlTemplate;
import org.t2framework.util.impl.UrlTemplateImpl;

import commons.Disposable;
import commons.Disposer;
import commons.exception.NoSuchComponentException;
import commons.exception.ResourceNotFoundRuntimeException;
import commons.meta.BeanDesc;
import commons.meta.ClassDesc;
import commons.meta.ConfigDesc;
import commons.meta.impl.ClassDescImpl;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.ResourceUtil;
import commons.util.StringUtil;
import commons.util.Traversal;
import commons.util.Reflections.ClassUtil;
import commons.util.Reflections.PackageUtil;

/**
 * Concrete class of org.t2framework.contexts.WebApplication.
 * 
 * @author shot
 * @see org.t2framework.contexts.WebApplication
 */
public class WebApplicationImpl implements WebApplication,
		Traversal.ClassHandler {

	protected final ContainerAdapter<?> adapter;

	protected FilterConfig config;

	protected ServletContext context;

	protected String encoding;

	protected final String[] rootPackages;

	protected final boolean eagerload;

	protected ConcurrentHashMap<String, PageDesc> pageDescMap = new ConcurrentHashMap<String, PageDesc>();

	@SuppressWarnings("unchecked")
	protected Map<Class, UrlTemplate> classCache = CollectionsUtil
			.newConcurrentHashMap();

	/**
	 * This will work from version 0.5.
	 */
	protected final boolean amfEnabled = false;

	protected final AnnotationResolverCreator resolverCreator;

	public WebApplicationImpl(ContainerAdapter<?> adapter,
			final FilterConfig config, final String encoding,
			final String[] rootPackages, final boolean eagerload,
			final AnnotationResolverCreator resolverCreator) {
		this.adapter = Assertion.notNull(adapter, "adapter");
		this.config = Assertion.notNull(config);
		this.context = Assertion.notNull(config.getServletContext());
		this.encoding = Assertion.notNull(encoding);
		this.rootPackages = Assertion.notNull(rootPackages);
		this.eagerload = eagerload;
		this.resolverCreator = Assertion.notNull(resolverCreator);
		traverse();
		setupDispose();
	}

	protected synchronized void traverse() {
		for (String rootPackage : rootPackages) {
			final String path = rootPackage.replace('.', '/');
			try {
				File rootDir = ResourceUtil.getBuildDir(path);
				Traversal.traverse(rootDir, rootPackage, this);
			} catch (ResourceNotFoundRuntimeException e) {
				throw new RootPackageNotFoundException(path);
			}
		}
	}

	@Override
	public WebContext createContext(final HttpServletRequest req,
			final HttpServletResponse res, final FilterChain filterChain) {
		final Request request = new RequestImpl(req, res, amfEnabled);
		final Response response = new ResponseImpl(res);
		final Session session = new SessionImpl(request);
		final Application application = new ApplicationImpl(context, config);
		final Chain chain = new ChainImpl(filterChain);
		ContextImpl context = new ContextImpl(request, response, application,
				session, chain);
		WebContext.set(context);
		return context;
	}

	@SuppressWarnings("unchecked")
	@Override
	public void processClass(String packageName, String shortClassName) {
		if (packageName == null || shortClassName == null) {
			return;
		}
		final Class<?> c = getPageClass(packageName, shortClassName);
		final int mod = c.getModifiers();
		if (c == null || c.isAnnotation() || c.isEnum()
				|| Modifier.isInterface(mod) || Modifier.isAbstract(mod)
				|| !Modifier.isPublic(mod)) {
			return;
		}
		final ClassDesc<?> cd = new ClassDescImpl(c);
		processPage(cd);
	}

	@SuppressWarnings("unchecked")
	protected void processPage(ClassDesc<?> cd) {
		if (!cd.hasConfigDesc(Page.class)) {
			return;
		}
		ConfigDesc pageConfig = cd.findConfigDesc(Page.class);
		final Page page = (Page) pageConfig.getAnnotation();
		if (page == null) {
			return;
		}
		final Class c = cd.getConcreteClass();
		String path = page.value();
		if (StringUtil.isEmpty(path)) {
			String name = ClassUtil.getShortClassName(c);
			path = StringUtil.decapitalize(name);
		}
		if (eagerload) {
			final ContainerAdapter adapter = getContainerAdapter();
			adapter.register(c);
		}
		final UrlTemplate template = new UrlTemplateImpl(path);
		classCache.put(c, template);
	}

	protected ContainerAdapter<?> getContainerAdapter() {
		return adapter;
	}

	protected void setupDispose() {
		Disposer.add(new Disposable() {
			@Override
			public void dispose() {
				config = null;
				context = null;
				pageDescMap.clear();
				classCache.clear();
			}
		});
	}

	@Override
	public Map<String, PageDesc> getPageDescMap() {
		if (pageDescMap.isEmpty()) {
			loadPageDescMap();
		}
		return pageDescMap;
	}

	@SuppressWarnings("unchecked")
	protected void loadPageDescMap() {
		for (Iterator<Class> itr = classCache.keySet().iterator(); itr
				.hasNext();) {
			Class key = itr.next();
			UrlTemplate template = classCache.get(key);
			final BeanDesc<?> beanDesc = adapter.getBeanDesc(key);
			if (beanDesc == null) {
				throw new NoSuchComponentException(key.getName());
			}
			final PageDesc pd = new PageDescImpl(PackageUtil
					.getPackageName(key), ClassUtil.getSimpleClassName(key),
					beanDesc, template, resolverCreator
							.getActionAnnotationSet(adapter));
			pageDescMap.putIfAbsent(key.getName(), pd);
		}
	}

	protected Class<?> getPageClass(final String packageName,
			String shortClassName) {
		String s = packageName.replace('/', '.');
		return ClassUtil.forNameNoException(s + "." + shortClassName);
	}

	@Override
	public String getEncoding() {
		return encoding;
	}

	@Override
	public String[] getRootPackages() {
		return rootPackages;
	}

}
