package org.t2framework.contexts.impl;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import org.t2framework.contexts.ActionMethodDesc;
import org.t2framework.contexts.PageDesc;
import org.t2framework.util.UrlTemplate;

import commons.meta.BeanDesc;
import commons.meta.BeanDescFactory;
import commons.meta.ClassDesc;
import commons.meta.MethodDesc;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.StringUtil;
import commons.util.Reflections.ClassUtil;
import commons.util.Reflections.PackageUtil;

/**
 * 
 * @author shot
 */
public class PageDescImpl implements PageDesc {

	protected String packageName;

	protected String pageName;

	protected MethodDesc defaultMethod;

	protected ClassDesc<?> pageClassDesc;

	protected List<Pattern> patterns = CollectionsUtil.newArrayList();

	protected ActionMethodDesc actionMethodDesc;

	protected UrlTemplate template;

	protected final Set<Class<? extends Annotation>> actionAnnotationSet;

	public PageDescImpl(final Class<?> clazz, final UrlTemplate template,
			Set<Class<? extends Annotation>> actionAnnotationSet) {
		this(PackageUtil.getPackageName(Assertion.notNull(clazz)), ClassUtil
				.getShortClassName(clazz), BeanDescFactory.getBeanDesc(clazz),
				Assertion.notNull(template), actionAnnotationSet);
	}

	public PageDescImpl(final String packageName, final String shortClassName,
			final BeanDesc<?> beanDesc, final UrlTemplate template,
			final Set<Class<? extends Annotation>> actionAnnotationSet) {
		Assertion.notNulls(beanDesc, packageName, shortClassName, template);
		this.packageName = Assertion.notNull(packageName);
		this.pageName = Assertion.notNull(shortClassName);
		this.template = template;
		this.actionAnnotationSet = Assertion.notNull(actionAnnotationSet);
		initActionMethodDesc(beanDesc);
	}

	protected void initActionMethodDesc(BeanDesc<?> beanDesc) {
		this.pageClassDesc = beanDesc.getClassDesc();
		this.actionMethodDesc = ActionMethodUtil.createActionMethodDesc(
				beanDesc, actionAnnotationSet);
		final MethodDesc defaultMethod = ActionMethodUtil
				.resolveDefaultMethodDesc(beanDesc);
		if (defaultMethod != null) {
			actionMethodDesc.addDefaultMethodDesc(defaultMethod);
			this.defaultMethod = defaultMethod;
		}
	}

	@Override
	public MethodDesc getDefaultMethodDesc() {
		return defaultMethod;
	}

	@Override
	public String getPackageName() {
		return packageName;
	}

	@Override
	public String getPageName() {
		return pageName;
	}

	@Override
	public ClassDesc<?> getPageClassDesc() {
		return pageClassDesc;
	}

	@Override
	public boolean match(final String url) {
		if (StringUtil.isEmpty(url)) {
			return false;
		}
		String pagePath = normalize(url);
		boolean match = template.match(pagePath);
		if (match) {
			return true;
		}
		if (template.getTemplatePath().equals(url)) {
			return true;
		}
		return url.equalsIgnoreCase(pageName);
	}

	protected String normalize(String url) {
		final String normalized = normalize0(url);
		final int pos = findPosition(normalized);
		return normalized.substring(0, pos);
	}

	protected int findPosition(String normalized) {
		int pos = normalized.length();
		final int count = StringUtil.counts(template.getTemplatePath(), '/');
		for (int i = 0, index = 0; i < normalized.length(); i++) {
			if (index == count + 1) {
				pos = i - 1;
				break;
			}
			char c = normalized.charAt(i);
			if (c == '/') {
				index++;
			}
		}
		return pos;
	}

	protected String normalize0(String url) {
		final String templatePath = template.getTemplatePath();
		final boolean templateStartSlash = templatePath.indexOf("/") == 0;
		final boolean urlStartSlash = url.indexOf("/") == 0;
		if (templateStartSlash) {
			url = urlStartSlash ? url : "/" + url;
		} else {
			url = urlStartSlash ? url.substring(1) : url;
		}

		final boolean templateEndSlash = templatePath.lastIndexOf("/") == templatePath
				.length() - 1;
		final boolean urlEndSlash = url.lastIndexOf("/") == url.length() - 1;
		if (templateEndSlash) {
			url = urlEndSlash ? url : url + "/";
		} else {
			url = urlEndSlash ? url.substring(0, url.length() - 1) : url;
		}
		return url;
	}

	@Override
	public ActionMethodDesc getActionMethodDesc() {
		return actionMethodDesc;
	}

	@Override
	public boolean hasDefaultMethodDesc() {
		return defaultMethod != null;
	}

	@Override
	public UrlTemplate getUrlTemplate() {
		return template;
	}

	@Override
	public String getPageTemplatePath() {
		return template.getTemplatePath();
	}

}
