package org.t2framework.contexts;

import java.util.Locale;

import org.t2framework.action.ActionContext;

import commons.Disposable;
import commons.Disposer;
import commons.util.Assertion;

/**
 * Base context class for T2, and it is stored at each thread using
 * ThreadLocal.WebContext is created when each request is accepted to T2 entry
 * point(usually expected T2Filter).
 * 
 * @author shot
 */
public abstract class WebContext implements
		Context<Request, Response, Application, Session> {

	protected static ThreadLocal<WebContext> pool = createThreadLocal();

	protected Locale defaultLocale;

	protected ActionContext actionContext;

	public WebContext() {
		this(Locale.getDefault());
	}

	public WebContext(final Locale defaultLocale) {
		Disposer.add(new Disposable() {
			@Override
			public void dispose() {
				pool.remove();
			}
		});
		this.defaultLocale = Assertion.notNull(defaultLocale);
	}

	public static void set(WebContext context) {
		pool.set(Assertion.notNull(context));
	}

	public static WebContext get() {
		return pool.get();
	}

	public static void clear() {
		pool.set(null);
	}

	private static <T> ThreadLocal<T> createThreadLocal() {
		return new ThreadLocal<T>();
	}

	public Locale getRequestLocale() {
		return getRequest().getLocale();
	}

	public Locale getDefaultLocale() {
		return this.defaultLocale;
	}

	public void forward(final String path) {
		Assertion.notNull(path);
		getRequest().forward(path);
	}

	public void redirect(final String path) {
		Assertion.notNull(path);
		getResponse().redirect(path);
	}

	public ActionContext getActionContext() {
		return actionContext;
	}

	public void setActionContext(ActionContext actionContext) {
		this.actionContext = Assertion.notNull(actionContext);
	}

}
