package org.t2framework.action.impl;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;

import org.t2framework.action.ActionContext;
import org.t2framework.action.ActionContextBuilder;
import org.t2framework.action.ActionInvoker;
import org.t2framework.action.PageDescFinder;
import org.t2framework.action.PageDescFinderFactory;
import org.t2framework.action.ParameterResolver;
import org.t2framework.adapter.ContainerAdapter;
import org.t2framework.contexts.PageDesc;
import org.t2framework.contexts.WebContext;
import org.t2framework.exception.ExceptionHandlerUnprocessingRuntimeException;
import org.t2framework.exception.NoParameterResolverRuntimeException;
import org.t2framework.handler.ExceptionHandlerChain;
import org.t2framework.handler.impl.ExceptionHandlerChainImpl;
import org.t2framework.navigation.NoOperation;
import org.t2framework.navigation.PassThrough;
import org.t2framework.plugin.PluginProcessor;
import org.t2framework.spi.AnnotationResolverCreator;
import org.t2framework.spi.Navigation;

import commons.exception.InvocationTargetRuntimeException;
import commons.meta.MethodDesc;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.Logger;

/**
 * Concrete class of ActionInvoker.
 * 
 * @author shot
 * 
 */
public class ActionInvokerImpl implements ActionInvoker {

	protected WebContext context;

	protected Map<String, PageDesc> pageDescMap;

	protected PageDesc targetPageDesc = null;

	@SuppressWarnings("unchecked")
	protected Class pageClass = null;

	protected ActionContextBuilder builder;

	protected Object lock = new Object();

	protected final ContainerAdapter<?> containerAdapter;

	protected final AnnotationResolverCreator resolverCreator;

	protected final PluginProcessor pluginProcessor;

	protected ExceptionHandlerChain exceptionHandlerChain;

	protected static Logger log = Logger.getLogger(ActionInvokerImpl.class);

	protected List<ParameterResolver> parameterResolvers = CollectionsUtil
			.emptyList();

	public ActionInvokerImpl(final WebContext context,
			final Map<String, PageDesc> pageDescMap,
			final ContainerAdapter<?> containerAdapter,
			final AnnotationResolverCreator resolverCreator,
			final PluginProcessor pluginProcessor) {
		this.context = context;
		this.pageDescMap = pageDescMap;
		this.containerAdapter = Assertion.notNull(containerAdapter);
		this.resolverCreator = Assertion.notNull(resolverCreator);
		this.pluginProcessor = Assertion.notNull(pluginProcessor);
		init();
	}

	protected synchronized void init() {
		final ActionContext actionContext = new ActionContextImpl(context);
		context.setActionContext(actionContext);
		final PageDescFinder pageDescFinder = PageDescFinderFactory
				.getPageDescFinder(context, pageDescMap);
		if (pageDescFinder != null) {
			PageDesc pd = pageDescFinder.find(context, pageDescMap);
			if (pd == null) {
				return;
			}
			log.log("WTDT0016", null, new Object[] { pd.getPageName() });
			this.targetPageDesc = pd;
			this.pageClass = targetPageDesc.getPageClassDesc()
					.getConcreteClass();
			actionContext.setTargetPageDesc(pd);
		}
		this.exceptionHandlerChain = new ExceptionHandlerChainImpl(
				this.containerAdapter.createExceptionHandlers());
		this.parameterResolvers = resolverCreator.createParameterResolvers(
				this.containerAdapter, this.context);
		this.builder = new ActionContextBuilderImpl(
				this.resolverCreator
						.createActionAnnotationResolvers(containerAdapter),
				this.resolverCreator
						.createDefaultActionAnnotationResolver(containerAdapter));
	}

	@Override
	public boolean match() {
		return targetPageDesc != null;
	}

	@Override
	public Navigation invoke() {
		Navigation ret = null;
		try {
			final ActionContext actionContext = context.getActionContext();
			if (actionContext == null) {
				return PassThrough.pass();
			}
			builder.build(actionContext);
			final MethodDesc actionMethod = actionContext.getTargetMethodDesc();
			if (actionMethod == null) {
				return PassThrough.pass();
			}
			final Object page = getPage();
			ret = invokeComponentCreated(page);
			if (isNavigateImmediatelly(ret)) {
				return ret;
			}
			final Object[] args = prepareArguments(actionContext, actionMethod);
			try {
				ret = beforeActionInvoke(actionContext, page, actionMethod,
						args);
				if (isNavigateImmediatelly(ret)) {
					return ret;
				}
				ret = invokeAction(actionContext, page, actionMethod, args);
			} finally {
				Navigation n = afterActionInvoke(actionContext, page,
						actionMethod, args);
				if (isNavigateImmediatelly(n)) {
					ret = n;
				}
			}
			if (ret == null) {
				// TODO error handler.
				ret = PassThrough.pass();
			}
		} catch (Throwable t) {
			Navigation n = null;
			try {
				n = exceptionHandlerChain.doChain(t, context);
			} catch (ExceptionHandlerUnprocessingRuntimeException e) {
				throw e;
			}
			if (n != null) {
				ret = n;
			} else {
				throw new ExceptionHandlerUnprocessingRuntimeException(t);
			}
		}
		return ret;
	}

	protected boolean isNavigateImmediatelly(Navigation navigation) {
		return navigation != null && navigation != NoOperation.INSTANCE;
	}

	protected Navigation invokeComponentCreated(Object page) {
		return pluginProcessor.invokeComponentCreated(context, page);
	}

	protected Navigation beforeActionInvoke(ActionContext actionContext,
			Object page, MethodDesc methodDesc, Object[] args) {
		return pluginProcessor.beforeActionInvoke(actionContext, methodDesc,
				page, args);
	}

	protected Navigation invokeAction(final ActionContext actionContext,
			final Object page, final MethodDesc methodDesc, Object[] args) {
		try {
			return (Navigation) methodDesc.invoke(page, args);
		} catch (Throwable t) {
			if (t instanceof RuntimeException) {
				throw (RuntimeException) t;
			} else {
				throw new IllegalStateException(t);
			}
		}
	}

	protected Navigation afterActionInvoke(ActionContext actionContext,
			Object page, MethodDesc methodDesc, Object[] args) {
		return pluginProcessor.afterActionInvoke(actionContext, methodDesc,
				page, args);
	}

	@SuppressWarnings("unchecked")
	protected Object[] prepareArguments(final ActionContext actionContext,
			final MethodDesc methodDesc)
			throws NoParameterResolverRuntimeException {
		final Class[] paramClasses = methodDesc.getParameterTypes();
		final Annotation[][] parameterAnnotations = methodDesc
				.getParameterAnnotations();
		Object[] args = new Object[paramClasses.length];
		resolve: for (int paramIndex = 0; paramIndex < paramClasses.length; paramIndex++) {
			final Annotation[] argAnnotations = parameterAnnotations[paramIndex];
			final Class argType = paramClasses[paramIndex];
			for (ParameterResolver resolver : parameterResolvers) {
				if (resolver.isTargetParameter(actionContext, methodDesc,
						paramIndex, argAnnotations, argType)) {
					args[paramIndex] = resolver.resolve(actionContext,
							methodDesc, paramIndex, argAnnotations, argType);
					continue resolve;
				}
			}
			throw new NoParameterResolverRuntimeException(argType);
		}
		return args;
	}

	protected Throwable unwrapIfInvocationTargetRuntimeException(Throwable t) {
		if (t instanceof InvocationTargetRuntimeException) {
			return t.getCause();
		}
		return t;
	}

	@SuppressWarnings("unchecked")
	protected Object getPage() {
		return containerAdapter.getComponent(pageClass);
	}

	public void setActionContextBuilder(ActionContextBuilder creator) {
		this.builder = Assertion.notNull(creator);
	}
}
