package org.t2framework.action.impl;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;

import org.t2framework.action.ActionAnnotationResolver;
import org.t2framework.action.ActionContext;
import org.t2framework.action.ActionContextBuilder;
import org.t2framework.exception.NoActionMethodFoundRuntimeException;

import commons.meta.ConfigDesc;
import commons.meta.MethodDesc;
import commons.util.Assertion;
import commons.util.CollectionsUtil;

/**
 * ActionContextBuilderImpl is an implementation class of ActionContextBuilder.
 * 
 * @author shot
 */
public class ActionContextBuilderImpl implements ActionContextBuilder {

	protected Map<Class<? extends Annotation>, ActionAnnotationResolver> annotationResolvers = CollectionsUtil
			.newHashMap();

	protected ActionAnnotationResolver defaultResolver;

	public ActionContextBuilderImpl(
			Map<Class<? extends Annotation>, ActionAnnotationResolver> annotationResolvers,
			ActionAnnotationResolver defaultResolver) {
		this.annotationResolvers = Assertion.notNull(annotationResolvers);
		this.defaultResolver = Assertion.notNull(defaultResolver);
	}

	@Override
	public void build(final ActionContext actionContext) {
		Assertion.notNull(actionContext);
		List<MethodDesc> matchList = null;
		int maxMatchCount = 0;
		search: for (String actionMethod : actionContext.getMethodNameList()) {
			final MethodDesc methodDesc = actionContext
					.getMethodDesc(actionMethod);
			actionContext.setTargetMethodDesc(methodDesc);
			actionContext.clearMatchCount();
			for (ConfigDesc configDesc : methodDesc.getConfigDescs()) {
				if (!configDesc.hasAnnotation()) {
					continue;
				}
				final Annotation annotation = configDesc.getAnnotation();
				final ActionAnnotationResolver resolver = findAnnotationResolver(annotation);
				if (resolver == null) {
					continue;
				}
				boolean match = setupActionContextByResolver(resolver,
						actionContext, annotation);
				if (!match) {
					continue search;
				}
			}
			final int matchCount = actionContext.getMatchCount();
			if (matchCount > maxMatchCount) {
				maxMatchCount = matchCount;
				matchList = CollectionsUtil.newArrayList();
				matchList.add(actionContext.getTargetMethodDesc());
			} else if (matchCount == maxMatchCount) {
				if (matchList == null) {
					continue search;
				}
				matchList.add(actionContext.getTargetMethodDesc());
			}
		}
		MethodDesc methodDesc = null;
		if (matchList != null) {
			if (matchList.size() >= 1) {
				methodDesc = matchList.get(0);
			} else {
				throw new NoActionMethodFoundRuntimeException();
			}
		} else {
			methodDesc = findDefaultMethodDesc(actionContext);
		}
		actionContext.setTargetMethodDesc(methodDesc);
	}

	protected boolean setupActionContextByResolver(
			ActionAnnotationResolver resolver, ActionContext actionContext,
			Annotation annotation) {
		resolver.preResolve(actionContext, annotation);
		boolean match = false;
		try {
			match = resolver.isMatch(actionContext, annotation);
			if (match) {
				actionContext.incrementMatchCount();
				resolver.resolve(actionContext, annotation);
			}
		} finally {
			resolver.postResolve(actionContext, annotation);
		}
		return match;
	}

	protected MethodDesc findDefaultMethodDesc(final ActionContext actionContext) {
		if (!actionContext.hasDefaultMethodDesc()) {
			throw new NoActionMethodFoundRuntimeException();
		}
		final Annotation annotation = actionContext.getDefaultAnnotation();
		defaultResolver.resolve(actionContext, annotation);
		return actionContext.getDefaultMethodDesc();
	}

	protected ActionAnnotationResolver findAnnotationResolver(
			Annotation annotation) {
		Assertion.notNull(annotation);
		return annotationResolvers.get(annotation.annotationType());
	}

	@Override
	public void setDefaultActionAnnotationResolver(
			final ActionAnnotationResolver defaultResolver) {
		this.defaultResolver = Assertion.notNull(defaultResolver);
	}

	@Override
	public void addActionAnnotationResolver(
			Class<? extends Annotation> annotationClass,
			ActionAnnotationResolver resolver) {
		annotationResolvers.put(annotationClass, resolver);
	}

}
