package lucy.config.stax;

import java.lang.annotation.ElementType;
import java.lang.reflect.Method;
import java.util.List;

import lucy.config.meta.ConfigurableBeanDesc;
import lucy.el.mvel.MVELExpression;
import lucy.exception.MethodNotFoundException;

import commons.el.Expression;
import commons.meta.InjectConfigDesc;
import commons.meta.MethodDesc;
import commons.meta.PropertyDesc;
import commons.meta.impl.MethodDescImpl;
import commons.meta.impl.MethodInjectConfigDesc;
import commons.meta.impl.PropertyDescImpl;
import commons.meta.impl.PropertyInjectConfigDesc;
import commons.util.CollectionsUtil;
import commons.util.JavaBeansUtil;
import commons.util.StringUtil;
import commons.util.Reflections.ClassUtil;
import commons.util.Reflections.MethodUtil;

public class InjectTagHandler implements XmlEventHandler {

	@SuppressWarnings("unchecked")
	@Override
	public void start(XmlEventContext context, Attributes attributes) {
		ConfigurableBeanDesc beanDesc = (ConfigurableBeanDesc) context.peek();
		Class<?> c = beanDesc.getComponentClass();
		String methodName = attributes.getValue("method");
		if (StringUtil.isEmpty(methodName)) {
			throw new IllegalStateException(
					"method attribute must not be blank");
		}
		String paramTypesStr = attributes.getValue("parameterTypes");
		Class<?>[] paramTypes = null;
		if (!StringUtil.isEmpty(paramTypesStr)) {
			String[] params = StringUtil.split(paramTypesStr, ",");
			paramTypes = new Class<?>[params.length];
			for (int i = 0; i < params.length; i++) {
				paramTypes[i] = ClassUtil.forName(params[i]);
			}
		}
		Method method = find(context, c, methodName, paramTypes);
		if (JavaBeansUtil.isSetMethod(methodName)) {
			beanDesc.addPropertyDesc(createPropertyDesc(c, method, paramTypes));
		} else {
			beanDesc.addMethodDesc(createMethodDesc(method, paramTypes));
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public void end(XmlEventContext context, String body) {
		ConfigurableBeanDesc beanDesc = (ConfigurableBeanDesc) context.peek();
		String source = body.trim();
		if (source.length() > 0) {
			Expression expression = new MVELExpression(source);
			beanDesc.getLatest().addExpression(expression);
		}
	}

	public MethodDesc createMethodDesc(final Method method,
			Class<?>[] paramTypes) {
		MethodDesc md = new MethodDescImpl(method);
		InjectConfigDesc injectDesc = new MethodInjectConfigDesc(md);
		injectDesc.setElementType(ElementType.METHOD);
		injectDesc.setParameterTypes(paramTypes);
		md.addConfigDesc(injectDesc);
		return md;
	}

	@SuppressWarnings("unchecked")
	public PropertyDesc<?> createPropertyDesc(final Class<?> componentClass,
			final Method writeMethod, Class<?>[] paramTypes) {
		String propertyName = JavaBeansUtil.getPropertyName(writeMethod
				.getName());
		final Method readMethod = getReadMethod(componentClass, writeMethod);
		Class<?> propertyType = null;
		if (readMethod != null) {
			propertyType = readMethod.getReturnType();
		} else {
			propertyType = JavaBeansUtil
					.findPropertyTypeFromWriteMethod(writeMethod);
		}
		PropertyDesc pd = new PropertyDescImpl(componentClass, propertyType,
				propertyName);
		pd.setWriteMethod(writeMethod);
		if (readMethod != null) {
			pd.setReadMethod(readMethod);
		}
		InjectConfigDesc injectDesc = new PropertyInjectConfigDesc(pd);
		injectDesc.setElementType(ElementType.METHOD);
		injectDesc.setParameterTypes(paramTypes);
		pd.addConfigDesc(injectDesc);
		return pd;
	}

	protected <T> Method getReadMethod(Class<T> componentClass,
			Method writeMethod) {
		return JavaBeansUtil.getReadMethodFromWriteMethod(componentClass,
				writeMethod);
	}

	protected Method find(XmlEventContext context, Class<?> componentClass,
			String methodName, Class<?>[] paramTypes) {
		List<Method> list = CollectionsUtil.newArrayList();
		for (Method m : componentClass.getMethods()) {
			if (methodName.equals(m.getName())) {
				list.add(m);
			}
		}
		if (list == null || list.isEmpty()) {
			throw new MethodNotFoundException(componentClass, methodName,
					context.currentEvent.getLocation().getLineNumber());
		}
		if (list.size() == 1) {
			return list.get(0);
		} else {
			if (paramTypes == null) {
				return list.get(0);
			} else {
				return MethodUtil.getDeclaredMethod(componentClass, methodName,
						paramTypes);
			}
		}
	}

}
