package lucy.config.stax;

import java.util.List;

import lucy.config.meta.ConfigurableBeanDesc;
import lucy.config.stax.LucyConfigTagHandler.LucyConfig;

import commons.meta.AspectConfigDesc;
import commons.meta.ClassDesc;
import commons.meta.InstanceConfigDesc;
import commons.meta.InstanceConfigDescFactory;
import commons.meta.impl.ClassDescImpl;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.StringUtil;
import commons.util.Traverser;
import commons.util.Reflections.ClassUtil;
import commons.util.Traversal.ClassHandler;

public class ComponentsTagHandler implements XmlEventHandler {

	@Override
	public void start(XmlEventContext context, Attributes attributes) {
		Components components = new Components();
		String packageName = attributes.getValue("package");
		if (!StringUtil.isEmpty(packageName)) {
			components.setPackageName(packageName);
		}
		String classPattern = attributes.getValue("classPattern");
		if (!StringUtil.isEmpty(classPattern)) {
			components.setClassPattern(classPattern);
		}
		String ignorePattern = attributes.getValue("ignorePattern");
		if (!StringUtil.isEmpty(ignorePattern)) {
			components.setIgnorePattern(ignorePattern);
		}
		String referenceClass = attributes.getValue("referenceClass");
		if (!StringUtil.isEmpty(referenceClass)) {
			components.setReferenceClass(ClassUtil.forName(referenceClass));
		}
		String instance = attributes.getValue("instance");
		if (StringUtil.isEmpty(instance)) {
			instance = InstanceConfigDescFactory.DEFAULT;
		}
		InstanceConfigDesc instanceConfig = InstanceConfigDescFactory
				.getInstanceConfigDesc(instance);
		components.setInstanceConfigDesc(instanceConfig);
		context.push(components);
	}

	@Override
	@SuppressWarnings("unchecked")
	public void end(XmlEventContext context, String body) {
		final Components components = (Components) context.pop();
		final LucyConfig config = (LucyConfig) context.peek();
		final Traverser traverser = new Traverser(components.getPackageName(),
				new String[] { components.getClassPattern() },
				new String[] { components.getIgnorePattern() }, components
						.getReferenceClass());
		traverser.traverse(new ClassHandler() {
			@Override
			public void processClass(String packageName, String shortClassName) {
				String className = packageName + "." + shortClassName;
				Class<?> c = ClassUtil.forNameNoException(className);
				if (c == null) {
					return;
				}
				ConfigurableBeanDesc beanDesc = new ConfigurableBeanDesc(c);
				beanDesc.setClassDesc(createClassDesc(c));
				addAspect(beanDesc);
				config.addBeanDesc(beanDesc);
			}

			protected void addAspect(ConfigurableBeanDesc beanDesc) {
				for (AspectConfigDesc aspectConfigDesc : components
						.getAspectConfigDesc()) {
					beanDesc.addAspectConfigDesc(aspectConfigDesc);
				}
			}

			protected ClassDesc<?> createClassDesc(Class<?> clazz) {
				ClassDesc cd = new ClassDescImpl(clazz);
				cd.addConfigDesc(components.getInstanceConfigDesc());
				return cd;
			}
		});
	}

	public static class Components implements AspectConfigDescAware {

		protected String packageName;

		protected String classPattern;

		protected String ignorePattern;

		protected Class<?> referenceClass;

		protected InstanceConfigDesc instanceConfigDesc;

		protected List<AspectConfigDesc> aspectConfigDescList = CollectionsUtil
				.newArrayList();

		public String getPackageName() {
			return packageName;
		}

		public void setPackageName(String packageName) {
			this.packageName = packageName;
		}

		public String getClassPattern() {
			return classPattern;
		}

		public void setClassPattern(String classPattern) {
			this.classPattern = classPattern;
		}

		public String getIgnorePattern() {
			return ignorePattern;
		}

		public void setIgnorePattern(String ignorePattern) {
			this.ignorePattern = ignorePattern;
		}

		public Class<?> getReferenceClass() {
			return referenceClass;
		}

		public void setReferenceClass(Class<?> referenceClass) {
			this.referenceClass = referenceClass;
		}

		public InstanceConfigDesc getInstanceConfigDesc() {
			return instanceConfigDesc;
		}

		public void setInstanceConfigDesc(InstanceConfigDesc instanceConfigDesc) {
			this.instanceConfigDesc = instanceConfigDesc;
		}

		@Override
		public void addAspectConfigDesc(AspectConfigDesc aspectConfigDesc) {
			aspectConfigDescList.add(Assertion.notNull(aspectConfigDesc));
		}

		public List<AspectConfigDesc> getAspectConfigDesc() {
			return aspectConfigDescList;
		}
	}
}
