package lucy.behavior;

import java.lang.reflect.Array;

import lucy.Lifecycle;
import lucy.Lucy;
import lucy.annotation.core.Inject;
import lucy.spi.AbstractBehavior;

import commons.annotation.ConfigurationTarget;
import commons.el.Expression;
import commons.meta.BeanDesc;
import commons.meta.ConfigDesc;
import commons.meta.ConfigDescContainer;
import commons.meta.ExpressionAware;
import commons.meta.InjectConfigDesc;
import commons.meta.MethodDesc;
import commons.meta.PropertyDesc;
import commons.util.Assertion;
import commons.util.AutoboxingUtil;
import commons.util.ConverterUtil;
import commons.util.Logger;

/**
 * Inject behavior is the behavior class which injects component using JavaBeans
 * property accessor or method.
 * 
 * @author shot
 */
@ConfigurationTarget( { Inject.class, InjectConfigDesc.class })
public class InjectBehavior extends AbstractBehavior {

	protected static Logger logger = Logger.getLogger(InjectBehavior.class);

	@SuppressWarnings("unchecked")
	public InjectBehavior(BeanDesc beanDesc, MethodDesc md, ConfigDesc cd) {
		super(beanDesc, md, cd);
	}

	@SuppressWarnings("unchecked")
	public InjectBehavior(BeanDesc beanDesc, PropertyDesc pd, ConfigDesc cd) {
		super(beanDesc, pd, cd);
	}

	/**
	 * <pre>
	 * Inject components to given parameter as t using Lucy.
	 * Execute method is called every time when @Inject is targeted in a component.
	 * For example, a component has two @Inject methods, so there should be two time execute called.
	 * 
	 * [Features of return value]
	 *  The return value must be injected components, which is targeted by @Inject 
	 *  or inject tag by xml file normally. If fail, no components injected.
	 *  
	 * [In case of fail]
	 *  In case of injecting fail, no components injected by this behavior.
	 *  
	 * [Etc]
	 *  It might work with static method injection, but not for sure.
	 * 
	 * </pre>
	 * 
	 * @param t
	 *            : target instance to be injected
	 * @param lucy
	 *            : IoC container
	 * @return t : object which may be injected some components
	 * @throws NullPointerException
	 *             in case of {@code Lucy} instance is {@code null}.
	 * @see lucy.Lucy
	 */
	@Override
	public <T> T execute(T t, Lucy lucy) {
		Assertion.notNull(lucy);
		injectPropertyDesc(t, lucy);
		injectMethodDesc(t, lucy);
		return t;
	}

	@SuppressWarnings("unchecked")
	protected <T> void injectPropertyDesc(T t, Lucy lucy) {
		PropertyDesc<Object> propertyDesc = getPropertyDesc();
		if (propertyDesc == null) {
			return;
		}
		Object o;
		final Class type = propertyDesc.getPropertyType();
		final boolean isArray = type.isArray();
		if (isArray) {
			Class<?>[] injectTypes = getTypesFromInject(propertyDesc);
			if (injectTypes != null) {
				Class<?> componentType = type.getComponentType();
				o = getInjectedComponents(injectTypes, componentType, lucy);
			} else {
				if (propertyDesc.hasExpression()) {
					Object[] evalResult = evaluateExpression(propertyDesc, lucy);
					if (evalResult != null && evalResult.length > 0) {
						o = convertTypes(evalResult[0], type.getComponentType());
					} else {
						o = null;
					}
				} else {
					o = lucy.getAll(type.getComponentType());
				}
			}
		} else {
			if (propertyDesc.hasExpression()) {
				Object[] evalResult = evaluateExpression(propertyDesc, lucy);
				if (evalResult != null && evalResult.length > 0) {
					o = evalResult[0];
				} else {
					o = null;
				}
			} else {
				o = lucy.get(type);
			}
		}
		if (o != null) {
			propertyDesc.setValue(t, o);
		} else {
			logger.debug("Injection value by property is null(type :"
					+ type.getName() + ((isArray ? "[]" : "")) + ").");
		}
	}

	@SuppressWarnings("unchecked")
	protected <T> void injectMethodDesc(T t, Lucy lucy) {
		MethodDesc methodDesc = getMethodDesc();
		if (methodDesc == null) {
			return;
		}
		Class[] types = getTypesFromInject(methodDesc);
		if (types == null) {
			types = methodDesc.getParameterTypes();
		}
		Object[] args = null;
		if (types.length == 1) {
			Object value = lucy.get(types[0]);
			if (value != null) {
				args = new Object[] { value };
			}
		} else {
			if (methodDesc.hasExpression()) {
				args = evaluateExpression(methodDesc, lucy);
			} else {
				args = getInjectedComponents(types, Object.class, lucy);
			}
		}
		if (args != null
				&& (args.length > 1 || (args.length == 1 && args[0] != null))) {
			methodDesc.invoke(t, args);
		} else {
			StringBuilder builder = new StringBuilder();
			builder.append("[");
			for (Class<?> type : types) {
				builder.append(type.getName());
				builder.append(", ");
			}
			builder.setLength(builder.length() - 2);
			builder.append("]");
			logger.debug("Injection value by method is null(type : "
					+ new String(builder) + ").");
		}

	}

	protected Object[] convertTypes(Object obj, Class<?> componentType) {
		final int length = Array.getLength(obj);
		Object[] values = (Object[]) Array.newInstance(componentType, length);
		for (int i = 0; i < length; i++) {
			Object convert = ConverterUtil.convert(Array.get(obj, i),
					componentType);
			System.out.println(convert);
			Array.set(values, i, convert);
		}
		return values;
	}

	@SuppressWarnings("unchecked")
	protected Object[] evaluateExpression(ExpressionAware aware, Lucy lucy) {
		Object[] objects = new Object[aware.getExpressionSize()];
		for (int i = 0; i < aware.getExpressionSize(); i++) {
			Expression expression = aware.getExpression(i);
			objects[i] = expression.evaluate(lucy);
		}
		return objects;
	}

	@SuppressWarnings("unchecked")
	protected Class[] getTypesFromInject(ConfigDescContainer desc) {
		Class[] types = null;
		for (int i = 0; i < desc.getConfigDescSize(); ++i) {
			ConfigDesc configDesc = desc.getConfigDesc(i);
			Class<?> type = configDesc.getType();
			if (type.equals(Inject.class)) {
				types = ((Inject) configDesc.getAnnotation()).types();
				if (types.length == 1 && types[0] == Object.class) {
					types = null;
				}
				return types;
			} else if (type.equals(InjectConfigDesc.class)) {
				return ((InjectConfigDesc) configDesc).getParameterTypes();
			}
		}
		return types;
	}

	@SuppressWarnings("unchecked")
	protected Object[] getInjectedComponents(Class[] classes,
			Class<?> componentType, Lucy lucy) {
		Object[] values = (Object[]) Array.newInstance(componentType,
				classes.length);
		for (int i = 0; i < classes.length; i++) {
			Object value = lucy.get(classes[i]);
			Array.set(values, i, AutoboxingUtil.getDefaultValueIfNull(
					classes[i], value));
		}
		return values;
	}

	@Override
	public Lifecycle getLifecycle() {
		return Lifecycle.COMPONENT_CREATED;
	}

}
