package lucy.aop.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import commons.exception.IORuntimeException;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.PatternUtil;
import commons.util.ReaderUtil;
import commons.util.StreamUtil;
import commons.util.SystemPropertyUtil;

public class MethodTemplateFactory {

	protected static final String SEP = SystemPropertyUtil
			.get("line.separator");

	private static final Pattern PATTERN = PatternUtil
			.getPattern("\\$\\{ *[a-zA-Z$_][a-zA-Z$_0-9]* *\\}");

	protected static MethodTemplate superMethod = MethodTemplate.defaultSuperMethod;

	protected static MethodTemplate method = MethodTemplate.defaultMethod;

	protected static MethodTemplate abstractmethod = MethodTemplate.defaultAbstractMethod;

	public static String getSuperMethodBody(final String methodName) {
		Map<String, String> context = CollectionsUtil.newHashMap();
		context.put("methodName", methodName);
		return MethodTemplateFactory.superMethod.parse(context);
	}

	public static String getAbstractMethodBody(final String methodName,
			final String interceptors) {
		Map<String, String> context = CollectionsUtil.newHashMap();
		context.put("methodName", methodName);
		context.put("interceptors", interceptors);
		return MethodTemplateFactory.abstractmethod.parse(context);
	}

	public static String getMethodBody(final String methodName,
			final String interceptors) {
		Map<String, String> context = CollectionsUtil.newHashMap();
		context.put("methodName", methodName);
		context.put("interceptors", interceptors);
		context.put("delegation", "methodName + \"$$\"");
		return MethodTemplateFactory.method.parse(context);
	}

	static MethodTemplate createMethodTemplate(final Class<?> clazz,
			final String templateName) {
		Assertion.notNulls(clazz, templateName);
		final InputStream is = clazz.getResourceAsStream(templateName);
		final BufferedReader reader = ReaderUtil.createBufferedReader(is);
		List<Token> tokenList = CollectionsUtil.newArrayList();
		try {
			String line;
			while (((line) = reader.readLine()) != null) {
				Matcher m = PATTERN.matcher(line);
				int start = 0;
				while (m.find()) {
					String value = line.substring(start, m.start());
					if (value.length() > 0) {
						tokenList.add(new StaticToken(value));
					}
					value = line.substring(m.start(), m.end());
					start = m.end();
					if (value.length() > 0) {
						tokenList.add(new DynamicToken(value));
					}
				}
				String value = line.substring(start);
				if (value.length() > 0) {
					tokenList.add(new StaticToken(value));
				}
				tokenList.add(new StaticToken(SEP));
			}
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally {
			StreamUtil.close(is);
		}
		final Token[] tokens = tokenList.toArray(new Token[tokenList.size()]);
		return new MethodTemplate() {

			@Override
			public String parse(Map<String, String> parameters) {
				StringBuilder builder = new StringBuilder();
				for (Token t : tokens) {
					builder.append(t.getValue(parameters));
				}
				return builder.toString();
			}

		};
	}

	private static final class DynamicToken implements Token {

		private String key;

		DynamicToken(String key) {
			this.key = key.substring(2, key.length() - 1).trim();
		}

		public String getValue(Map<String, String> parameters) {
			return parameters.get(this.key);
		}

		@Override
		public String toString() {
			return "${" + this.key + "}";
		}
	}

	private static final class StaticToken implements Token {

		private String value;

		StaticToken(String value) {
			this.value = value;
		}

		public String getValue(Map<String, String> parameters) {
			return this.value;
		}

		@Override
		public String toString() {
			return this.value;
		}
	}

	private static interface Token {

		String getValue(Map<String, String> parameters);
	}

}
