package lucy.aop.impl;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import commons.aop.spi.Interceptor;
import commons.aop.spi.Invocation;
import commons.util.Assertion;
import commons.util.AutoboxingUtil;
import commons.util.CollectionsUtil;
import commons.util.Reflections.MethodUtil;

public class InvocationImpl implements Invocation<Method> {

	protected Method method;

	protected Method delegation;

	protected Object[] args;

	protected Object instance;

	protected List<Interceptor> interceptors = CollectionsUtil.emptyList();

	protected int count = 0;

	public InvocationImpl(List<Interceptor> interceptors, Method method,
			Method delegation, Object[] args, Class<?> clazz, Object instance) {
		Assertion.notNull(clazz);
		this.interceptors = Assertion.notNull(interceptors);
		this.method = Assertion.notNull(method);
		this.delegation = Assertion.notNull(delegation);
		this.args = Assertion.notNulls(args);
		this.instance = Assertion.notNull(instance);
	}

	public InvocationImpl(
			Map<AccessibleObject, List<Interceptor>> interceptorMap,
			Method method, Method delegation, Object[] args, Class<?> clazz,
			Object instance) {
		Assertion.notNull(interceptorMap);
		Assertion.notNull(clazz);
		Class<?> c = getOriginalClass(clazz);
		final Method m = MethodUtil.getDeclaredMethod(c, method.getName(),
				method.getParameterTypes());
		this.interceptors = interceptorMap.get(m);
		this.method = Assertion.notNull(method);

		// Do not check null because if original class is an abstract class or
		// interface, there is no delegation method.
		this.delegation = delegation;
		this.args = Assertion.notNulls(args);
		this.instance = Assertion.notNull(instance);
	}

	/**
	 * Get original class even if enhanced.
	 * 
	 * @param clazz
	 * @return
	 */
	protected Class<?> getOriginalClass(Class<?> clazz) {
		if (isInterface(clazz)) {
			return clazz.getInterfaces()[0];
		} else {
			return clazz.getSuperclass();
		}
	}

	/**
	 * Return true if given enhanced clazz is originally interface.
	 * 
	 * @param clazz
	 * @return boolean original class is an interface or not.
	 */
	protected boolean isInterface(Class<?> clazz) {
		return (clazz.getSuperclass() == Object.class && clazz.getInterfaces().length > 0);
	}

	@Override
	public Object[] getArguments() {
		return args;
	}

	@Override
	public Object getThis() {
		return instance;
	}

	@Override
	public Method getType() {
		return method;
	}

	protected Lock lock = new ReentrantLock();

	/**
	 * doProceed() is called from enhanced method.
	 * 
	 * @return
	 * @throws Throwable
	 */
	public Object doProceed() throws Throwable {
		lock.lock();
		try {
			return proceed();
		} finally {
			count = 0;
			lock.unlock();
		}
	}

	@Override
	public Object proceed() throws Throwable {
		Object result = null;
		boolean invoked = false;
		if (count < interceptors.size()) {
			Interceptor interceptor = interceptors.get(count++);
			result = interceptor.intercept(this);
			invoked = true;
		}
		if (!invoked && delegation != null) {
			result = delegation.invoke(instance, args);
		}
		if (result != null) {
			return result;
		}
		Class<?> returnType = method.getReturnType();
		if (returnType.isPrimitive()) {

			return AutoboxingUtil.getDefaultPrimitiveValue(returnType);
		}
		return null;
	}

}
