package lucy.aop.impl;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtField;
import javassist.Modifier;
import lucy.aop.AspectDesc;
import lucy.aop.CannotCompileRuntimeException;
import lucy.aop.ClassPoolUtil;
import lucy.aop.InterceptorDesc;
import lucy.aop.InterceptorDescStrategy;
import lucy.aop.JavassistUtil;
import lucy.aop.LucyClassPool;

import commons.aop.spi.Interceptor;
import commons.util.Assertion;
import commons.util.Reflections.ClassLoaderUtil;

public class EnhancedClassBuilderImpl<T> extends
		AbstractEnhancedClassBuilder<T> {

	public EnhancedClassBuilderImpl(InterceptorDescStrategy strategy) {
		super(strategy);
	}

	@Override
	protected CtClass makeCtClass(AspectDesc aspectDesc) {
		return ClassEnhanceHelper.makeCtClass(aspectDesc);
	}

	@SuppressWarnings("unchecked")
	@Override
	protected Class<T> generateClass(final AspectDesc aspectDesc,
			CtClass enhancedCtClass) {
		Assertion.notNulls(aspectDesc, enhancedCtClass);
		final ClassLoader classLoader = ClassLoaderUtil
				.getClassLoader(aspectDesc.getOriginalClass());
		try {
			return (Class<T>) JavassistUtil.toClass(enhancedCtClass,
					classLoader);
		} finally {
			enhancedCtClass.detach();
			enhancedCtClass = null;
		}
	}

	@Override
	protected void createInterceptorsField(String interceptorsName,
			Class<?> originalClass, CtClass targetClass,
			Map<AccessibleObject, List<Interceptor>> interceptorMap) {
		LucyClassPool pool = ClassPoolUtil.getClassPool(originalClass);
		CtClass ctClass = JavassistUtil.getCtClass(pool, interceptorMap
				.getClass());
		try {
			CtField field = new CtField(ctClass, interceptorsName, targetClass);
			field.setModifiers(javassist.Modifier.PUBLIC
					| javassist.Modifier.STATIC);
			targetClass.addField(field);
		} catch (CannotCompileException e) {
			throw new CannotCompileRuntimeException(e);
		}
	}

	@Override
	protected void createInvokeSuperMethod(AspectDesc aspectDesc,
			CtClass enhancedCtClass,
			Collection<InterceptorDesc> interceptorDescList) {
		for (InterceptorDesc id : Assertion.notNull(interceptorDescList)) {
			final Method method = id.getMethod();
			if (method == null || Modifier.isAbstract(method.getModifiers())) {
				continue;
			}
			final String methodName = method.getName();
			final String generatedName = generateMethodName(methodName);
			final String body = MethodTemplateFactory
					.getSuperMethodBody(methodName);
			ClassEnhanceHelper.addCtNewMethod(aspectDesc, enhancedCtClass,
					method, generatedName, body);
		}
	}

	protected String generateMethodName(String methodName) {
		return methodName + "$$";
	}

	@Override
	protected String generateInterceptorsFieldName(Class<?> clazz) {
		String name = "$$INTERCEPTORS$$";
		int i = 0;
		while (true) {
			try {
				clazz.getDeclaredField(name);
			} catch (NoSuchFieldException e) {
				try {
					clazz.getField(name);
				} catch (NoSuchFieldException ex) {
					return name;
				}
			}
			name = name + Integer.toString(i++);
		}
	}

	@Override
	protected void createTargetMethod(AspectDesc aspectDesc,
			CtClass enhancedCtClass,
			Map<AccessibleObject, InterceptorDesc> map, String interceptorsName) {
		for (InterceptorDesc id : Assertion.notNull(map).values()) {
			final Method method = id.getMethod();
			if (method == null) {
				continue;
			}
			String methodName = method.getName();
			String body = null;
			final boolean isAbstract = Modifier.isAbstract(method
					.getModifiers());
			if (isAbstract) {
				body = MethodTemplateFactory.getAbstractMethodBody(methodName,
						interceptorsName);
			} else {
				body = MethodTemplateFactory.getMethodBody(methodName,
						interceptorsName);
			}
			ClassEnhanceHelper.addCtNewMethod(aspectDesc, enhancedCtClass,
					method, methodName, body);
		}
	}

}
