package lucy.aop.impl;

import java.util.List;

import lucy.aop.AspectDesc;
import lucy.aop.EnhancedClassBuilder;
import lucy.aop.EnhancedClassCache;
import lucy.aop.InterceptorDescStrategy;

import commons.aop.AopEngine;
import commons.aop.spi.Aspect;
import commons.util.Assertion;
import commons.util.Logger;
import commons.util.Reflections.ClassUtil;

public class AopEngineImpl implements AopEngine {

	protected InterceptorDescStrategy strategy;

	protected EnhancedClassCache classCache;

	protected static Logger logger = Logger.getLogger(AopEngineImpl.class);

	public AopEngineImpl() {
		strategy = new InterceptorDescStrategyImpl();
		classCache = new SimpleEnahncedClassCacheImpl();
	}

	public AopEngineImpl(InterceptorDescStrategy strategy,
			EnhancedClassCache cache) {
		this.strategy = Assertion.notNull(strategy);
		this.classCache = Assertion.notNull(cache);
	}

	@Override
	public void startup() {
	}

	@Override
	public <T> T createEnhancedClassInstance(Class<? extends T> originalClass,
			List<Aspect> aspects) {
		Class<? extends T> enhancedClass = classCache.get(originalClass);
		if (enhancedClass == null) {
			logger.debug("Starts enhancing to " + originalClass.getName());
			enhancedClass = buildEnhancedClass(originalClass, aspects);
		}
		return ClassUtil.newInstance(enhancedClass);
	}

	@Override
	public <T> Class<? extends T> getEnhancedClass(
			Class<? extends T> originalClass, List<Aspect> aspects) {
		Class<? extends T> enhancedClass = classCache.get(originalClass);
		if (enhancedClass == null) {
			enhancedClass = buildEnhancedClass(originalClass, aspects);
			classCache.cache(originalClass, enhancedClass);
		}
		return enhancedClass;
	}

	protected synchronized <T> Class<T> buildEnhancedClass(
			Class<? extends T> originalClass, List<Aspect> aspects) {
		AspectDesc aspectDesc = AspectDescFactory.createAspectDesc(
				originalClass, aspects);
		final EnhancedClassBuilder<T> builder = createBuilder(strategy);
		Class<T> enhancedClass = builder.build(aspectDesc);
		classCache.cache(originalClass, enhancedClass);
		return enhancedClass;
	}

	protected <T> EnhancedClassBuilder<T> createBuilder(
			InterceptorDescStrategy strategy) {
		return new EnhancedClassBuilderImpl<T>(strategy);
	}

	@Override
	public void shutdown() {
		classCache.clear();
		classCache = null;
		strategy = null;
	}

}
