package lucy.aop.impl;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javassist.CtClass;
import lucy.aop.AspectDesc;
import lucy.aop.EnhancedClassBuilder;
import lucy.aop.InterceptorDesc;
import lucy.aop.InterceptorDescStrategy;

import commons.aop.spi.Interceptor;
import commons.meta.BeanDesc;
import commons.meta.BeanDescFactory;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.Reflections.FieldUtil;

public abstract class AbstractEnhancedClassBuilder<T> implements
		EnhancedClassBuilder<T> {

	protected InterceptorDescStrategy strategy;

	public AbstractEnhancedClassBuilder(InterceptorDescStrategy strategy) {
		setInterceptorDescStrategy(strategy);
	}

	public Class<T> build(final AspectDesc aspectDesc) {
		Assertion.notNull(aspectDesc);
		Map<AccessibleObject, InterceptorDesc> map = createInterceptorDescMap(aspectDesc);

		Class<?> originalClass = aspectDesc.getOriginalClass();
		CtClass ctClass = makeCtClass(aspectDesc);
		createInvokeSuperMethod(aspectDesc, ctClass, map.values());

		final Map<AccessibleObject, List<Interceptor>> interceptorMap = CollectionsUtil
				.newHashMap();
		toInterceptorMap(map, interceptorMap);
		String interceptorsName = generateInterceptorsFieldName(originalClass);
		createInterceptorsField(interceptorsName, originalClass, ctClass,
				interceptorMap);

		createTargetMethod(aspectDesc, ctClass, map, interceptorsName);

		Class<T> generatedClass = generateClass(aspectDesc, ctClass);

		final Field field = FieldUtil
				.getField(generatedClass, interceptorsName);
		FieldUtil.set(field, null, interceptorMap);
		return generatedClass;
	}

	protected abstract void createInterceptorsField(String interceptorsName,
			Class<?> originalClass, CtClass ctClass,
			Map<AccessibleObject, List<Interceptor>> interceptorMap);

	protected void toInterceptorMap(
			Map<AccessibleObject, InterceptorDesc> from,
			Map<AccessibleObject, List<Interceptor>> to) {
		for (AccessibleObject ao : Assertion.notNull(from).keySet()) {
			InterceptorDesc desc = from.get(ao);
			to.put(ao, desc.getInterceptors());
		}
	}

	@SuppressWarnings("unchecked")
	protected Map<AccessibleObject, InterceptorDesc> createInterceptorDescMap(
			final AspectDesc aspectDesc) {
		final BeanDesc<T> beanDesc = (BeanDesc<T>) BeanDescFactory
				.getBeanDesc(aspectDesc.getOriginalClass());
		return strategy.assemble(beanDesc, aspectDesc);
	}

	@Override
	public void setInterceptorDescStrategy(InterceptorDescStrategy strategy) {
		this.strategy = Assertion.notNull(strategy);
	}

	protected abstract Class<T> generateClass(final AspectDesc aspectDesc,
			CtClass enhancedCtClass);

	protected abstract CtClass makeCtClass(AspectDesc aspectDesc);

	protected abstract void createInvokeSuperMethod(AspectDesc aspectDesc,
			CtClass enhancedCtClass,
			Collection<InterceptorDesc> interceptorDescList);

	protected abstract String generateInterceptorsFieldName(Class<?> clazz);

	protected abstract void createTargetMethod(AspectDesc aspectDesc,
			CtClass enhancedCtClass,
			Map<AccessibleObject, InterceptorDesc> map, String interceptorsName);

}