package lucy.aop;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.CtNewConstructor;
import javassist.NotFoundException;
import javassist.bytecode.ConstPool;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.AnnotationMemberValue;
import javassist.bytecode.annotation.ArrayMemberValue;
import javassist.bytecode.annotation.BooleanMemberValue;
import javassist.bytecode.annotation.ByteMemberValue;
import javassist.bytecode.annotation.CharMemberValue;
import javassist.bytecode.annotation.ClassMemberValue;
import javassist.bytecode.annotation.DoubleMemberValue;
import javassist.bytecode.annotation.FloatMemberValue;
import javassist.bytecode.annotation.IntegerMemberValue;
import javassist.bytecode.annotation.LongMemberValue;
import javassist.bytecode.annotation.MemberValue;
import javassist.bytecode.annotation.ShortMemberValue;
import javassist.bytecode.annotation.StringMemberValue;

import commons.exception.ClassNotFoundRuntimeException;
import commons.util.Assertion;
import commons.util.Reflections.ClassUtil;

public class JavassistUtil {

	public static CtClass getCtClass(final ClassPool pool, final Class<?> clazz) {
		Assertion.notNulls(pool, clazz);
		return getCtClass(pool, ClassUtil.getClassName(clazz));
	}

	public static CtClass getCtClass(final ClassPool pool, final String name) {
		Assertion.notNulls(pool, name);
		try {
			return pool.get(name);
		} catch (NotFoundException e) {
			throw new CtClassNotFoundRuntimeException(e, name);
		}
	}

	public static CtMethod getCtMethod(CtClass ctClass, Method method) {
		Assertion.notNull(ctClass, "ctClass");
		Assertion.notNull(method, "method");
		return getCtMethod(ctClass, method.getName());
	}

	public static CtMethod getCtMethod(CtClass ctClass, String methodName) {
		Assertion.notNull(ctClass, "ctClass");
		Assertion.notNull(methodName, "methodName");
		try {
			return ctClass.getDeclaredMethod(methodName);
		} catch (NotFoundException e) {
			throw new CtMethodNotFoundRuntimeException(e, methodName);
		}
	}

	public static final Class<?> toClass(CtClass ctClass) {
		try {
			return ctClass.toClass();
		} catch (CannotCompileException e) {
			throw new CannotCompileRuntimeException(e);
		}
	}

	public static final Class<?> toClass(CtClass ctClass, ClassLoader loader) {
		try {
			return ctClass.toClass(loader, null);
		} catch (CannotCompileException e) {
			throw new CannotCompileRuntimeException(e.getCause());
		}
	}

	public static void addDefaultConstructor(final CtClass ctClass) {
		Assertion.notNull(ctClass);
		try {
			ctClass
					.addConstructor(CtNewConstructor
							.defaultConstructor(ctClass));
		} catch (CannotCompileException e) {
			throw new CannotCompileRuntimeException(e);
		}
	}

	public static CtClass getAndRename(ClassPool classPool, String orgName,
			String newName) {
		Assertion.notNulls(classPool, orgName, newName);
		try {
			return classPool.getAndRename(orgName, newName);
		} catch (NotFoundException e) {
			throw new CtClassNotFoundRuntimeException(e, orgName);
		}
	}

	public static CtClass makeCtClass(ClassPool pool, String name,
			Class<?> superClass) {
		return makeCtClass(pool, name, getCtClass(pool, superClass));
	}

	public static CtClass makeCtClass(ClassPool pool, String name,
			CtClass superClass) {
		return pool.makeClass(name, superClass);
	}

	public static CtClass[] getCtClassArray(final ClassPool classPool,
			final Class<?>[] classes) {
		if (classes == null) {
			return null;
		}
		final CtClass[] result = new CtClass[classes.length];
		for (int i = 0; i < result.length; ++i) {
			result[i] = getCtClass(classPool, classes[i]);
		}
		return result;
	}

	public static MemberValue getMemberValue(Class<?> returnType, Object value,
			ConstPool constPool) {
		Assertion.notNull(returnType, "returnType");
		Assertion.notNull(constPool, "constPool");
		Class<?> type = ClassUtil.toWrapperClass(returnType);
		if (type == String.class) {
			return new StringMemberValue(String.class.cast(value), constPool);
		} else if (type == Annotation.class) {
			return new AnnotationMemberValue(Annotation.class.cast(value),
					constPool);
		} else if (type == Integer.class) {
			return new IntegerMemberValue(Integer.class.cast(value), constPool);
		} else if (type == Array.class) {
			return new ArrayMemberValue(MemberValue.class.cast(value),
					constPool);
		} else if (type == Boolean.class) {
			return new BooleanMemberValue(Boolean.class.cast(value), constPool);
		} else if (type == Byte.class) {
			return new ByteMemberValue(Byte.class.cast(value), constPool);
		} else if (type == Character.class) {
			return new CharMemberValue(Character.class.cast(value), constPool);
		} else if (type == Class.class) {
			if (value.getClass() == Class.class) {
				return new ClassMemberValue(Class.class.cast(value).getName(),
						constPool);
			} else {
				return new ClassMemberValue(String.class.cast(value), constPool);
			}
		} else if (type == Double.class) {
			return new DoubleMemberValue(Double.class.cast(value), constPool);
		} else if (type == Enum.class) {
			throw new UnsupportedOperationException();
		} else if (type == Float.class) {
			return new FloatMemberValue(Float.class.cast(value), constPool);
		} else if (type == Long.class) {
			return new LongMemberValue(Long.class.cast(value), constPool);
		} else if (type == Short.class) {
			return new ShortMemberValue(Short.class.cast(value), constPool);
		}
		return null;
	}

	public static java.lang.annotation.Annotation getAnnotation(CtClass ctClass) {
		Assertion.notNull(ctClass);
		Object[] array = null;
		try {
			array = ctClass.getAnnotations();
		} catch (ClassNotFoundException e) {
			throw new ClassNotFoundRuntimeException(e);
		}
		return (java.lang.annotation.Annotation) ((array.length > 0) ? array[0]
				: null);
	}

	public static java.lang.annotation.Annotation getAnnotation(
			CtMethod ctMethod) {
		Assertion.notNull(ctMethod);
		Object[] array = null;
		try {
			array = ctMethod.getAnnotations();
		} catch (ClassNotFoundException e) {
			throw new ClassNotFoundRuntimeException(e);
		}
		return (java.lang.annotation.Annotation) ((array.length > 0) ? array[0]
				: null);
	}

	public static void addCtMethod(final CtClass ctClass,
			final CtMethod ctMethod) {
		Assertion.notNull(ctClass);
		Assertion.notNull(ctMethod);
		try {
			ctClass.addMethod(ctMethod);
		} catch (CannotCompileException e) {
			throw new CannotCompileRuntimeException(e.getCause());
		}
	}

}
