package lucy;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import lucy.exception.CyclicReferenceRuntimeException;
import lucy.exception.LucyComponentRegistrarNotFoundRuntimeException;
import lucy.exception.NoComponentAnnotationException;
import lucy.spi.ComponentBuilder;
import lucy.spi.LucyConfiguration;

import commons.annotation.core.Component;
import commons.exception.NoSuchComponentException;
import commons.meta.BeanDesc;
import commons.meta.BeanDescFactory;
import commons.meta.NameDesc;
import commons.meta.PackageDesc;
import commons.meta.impl.NameDescImpl;
import commons.util.AnnotationUtil;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.Logger;
import commons.util.Reflections.PackageUtil;

/**
 * IoC implementation.
 * 
 * @author shot
 */
public class LucyImpl implements Lucy {

	protected Map<Class<?>, BeanDescBag<?>> beanDescBagMap = CollectionsUtil
			.newConcurrentHashMap();

	protected List<BeanDesc<?>> namedBeanDescList = CollectionsUtil
			.newArrayList();

	protected List<PackageDesc> packageDescList = CollectionsUtil
			.newArrayList();

	protected ConfigDescBinder binder;

	protected Map<BeanDesc<?>, ConfigBindResult> bindMap = CollectionsUtil
			.newConcurrentHashMap();

	protected LucyConfiguration lucyConf;

	protected BeanDescStrategy beanDescStrategy;

	private ThreadLocal<BuildingPath> buildingPaths = new ThreadLocal<BuildingPath>();

	protected static Logger log = Logger.getLogger(LucyImpl.class);

	public LucyImpl() {
		this(new DefaultLucyConfiguration());
	}

	public LucyImpl(final LucyConfiguration conf) {
		Assertion.notNull(conf);
		this.lucyConf = conf;
	}

	public void init() {
		this.binder = lucyConf.loadConfigDescBinder();
		this.beanDescStrategy = lucyConf.getBeanDescStrategy(this);
	}

	@SuppressWarnings("unchecked")
	public <T> BeanDesc<T> getBeanDesc(Object identifier) {
		Assertion.notNull(identifier);
		BeanDesc<?> beanDesc = null;
		if (namedBeanDescList == null || namedBeanDescList.size() == 0) {
			return null;
		}
		for (BeanDesc<?> bd : namedBeanDescList) {
			NameDesc<?> nameDesc = bd.getNameDesc();
			if (nameDesc.isIdentified(identifier)) {
				beanDesc = bd;
				break;
			}
		}
		return (BeanDesc<T>) beanDesc;
	}

	@SuppressWarnings("unchecked")
	public <T> T get(Object identifier) {
		final BeanDesc<?> beanDesc = getBeanDesc(identifier);
		return (T) buildComponent(identifier, beanDesc);
	}

	@SuppressWarnings("unchecked")
	public <T> BeanDesc<T> getBeanDesc(Class<? super T> key) {
		final BeanDescBag beanDescBag = beanDescBagMap.get(key);
		if (beanDescBag != null) {
			return beanDescBag.getBeanDesc();
		} else {
			return null;
		}
	}

	public <T> T get(Class<? super T> key) {
		final BeanDesc<T> beanDesc = getBeanDesc(key);
		return buildComponent(key, beanDesc);
	}

	@SuppressWarnings("unchecked")
	protected <T> T buildComponent(final Object key, BeanDesc<T> beanDesc) {
		if (beanDesc == null) {
			// TODO it works without (T) cast at Eclipse, but not at maven2, so
			// simply cast it.
			return (T) beanDescStrategy.handleNoBeanDesc(key);
		}

		BuildingPath buildingPath = getBuildingPath();
		if (buildingPath.contains(beanDesc)) {
			if (beanDesc.isSingleton()) {
				return (T) beanDesc.getComponent();
			} else if (!buildingPath.isSingletonPresent()) {
				throw new CyclicReferenceRuntimeException();
			}
		}

		buildingPath.add(beanDesc);
		try {
			final T t = (T) beanDesc.getComponent();
			if (t != null) {
				if (LucyUtil.isSimpleType(t.getClass())) {
					return t;
				} else {
					bind(beanDesc);
				}
			}
			ConfigBindResult result = bindMap.get(beanDesc);
			if (result == null) {
				throw new NoSuchComponentException(key);
			}
			return createComponentBuilder(beanDesc).build(t, result);
		} finally {
			buildingPath.remove(beanDesc);
			if (buildingPath.isEmpty()) {
				removeBuildingPath();
			}
		}
	}

	BuildingPath getBuildingPath() {
		BuildingPath buildingPath = buildingPaths.get();
		if (buildingPath == null) {
			buildingPath = new BuildingPath();
			buildingPaths.set(buildingPath);
		}
		return buildingPath;
	}

	void removeBuildingPath() {
		buildingPaths.set(null);
	}

	protected <T> ComponentBuilder createComponentBuilder(BeanDesc<T> beanDesc) {
		return lucyConf.createBuilder(beanDesc, this);
	}

	public <T> Lucy register(Class<? extends T> componentClass) {
		Assertion.notNull(componentClass);
		assertComponentAnnotation(componentClass);
		if (BeanDescFactory.contains(componentClass)) {
			return this;
		}
		final BeanDesc<T> beanDesc = BeanDescFactory
				.getBeanDesc(componentClass);
		register(beanDesc);
		return this;
	}

	@Override
	public <T> Lucy register(Class<? extends T> componentClass,
			Object identifier) {
		Assertion.notNulls(componentClass, identifier);
		assertComponentAnnotation(componentClass);
		final BeanDesc<T> beanDesc = BeanDescFactory
				.createBeanDesc(componentClass);
		register(beanDesc, identifier);
		return this;
	}

	@Override
	public <T> Lucy register(BeanDesc<? extends T> beanDesc, Object identifier) {
		Assertion.notNull(beanDesc, "beanDesc");
		Assertion.notNull(identifier, "identifier");
		final NameDesc<T> nameDesc = createNameDesc(identifier);
		registerWithNameDesc(beanDesc, nameDesc);
		return this;
	}

	@SuppressWarnings("unchecked")
	protected <T> Lucy registerWithNameDesc(BeanDesc<? extends T> beanDesc,
			NameDesc<T> nameDesc) {
		Assertion.notNull(beanDesc, "beanDesc");
		Assertion.notNull(nameDesc, "nameDesc");
		beanDesc.setNameDesc((NameDesc) nameDesc);
		register(beanDesc);
		log.debug("register component :"
				+ beanDesc.getConcreteClass().getName());
		return this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> Lucy register(T t) {
		Assertion.notNull(t);
		Class<T> componentClass = (Class<T>) t.getClass();
		assertComponentAnnotation(componentClass);
		if (BeanDescFactory.contains(componentClass)) {
			return this;
		}
		BeanDesc<T> beanDesc = BeanDescFactory.getBeanDesc(t);
		register(beanDesc);
		return this;
	}

	@Override
	public <T> Lucy register(T t, Object identifier) {
		Assertion.notNull(t);
		assertComponentAnnotation(t.getClass());
		final BeanDesc<T> beanDesc = BeanDescFactory.createBeanDesc(t);
		final NameDesc<T> nameDesc = createNameDesc(identifier);
		beanDesc.setNameDesc(nameDesc);
		namedBeanDescList.add(beanDesc);
		register(beanDesc);
		return this;
	}

	public <T> Lucy register(final BeanDesc<? extends T> beanDesc) {
		final List<Class<?>> dependencyClassesList = beanDesc.getClassDesc()
				.getDependencyClassesList();
		registerBeanDescByClasses(beanDesc, dependencyClassesList);
		if (beanDesc.hasName()) {
			namedBeanDescList.add(beanDesc);
		}
		bind(beanDesc);
		return this;
	}

	protected <T> void bind(final BeanDesc<? extends T> beanDesc) {
		if (bindMap.containsKey(beanDesc)) {
			return;
		}
		final ConfigBindResult bindResult = binder.bind(beanDesc);
		if (!bindResult.isInitialized()) {
			bindResult.initializeBehaviors();
		}
		bindMap.put(beanDesc, bindResult);
	}

	@SuppressWarnings("unchecked")
	protected <T> void registerBeanDescByClasses(
			final BeanDesc<? extends T> beanDesc,
			List<Class<?>> dependencyClassesList) {
		for (Class<?> c : dependencyClassesList) {
			BeanDescBag bag = beanDescBagMap.get(c);
			if (bag == null) {
				bag = new BeanDescBag(c, beanDescStrategy);
				beanDescBagMap.put(c, bag);
			}
			bag.add(beanDesc);
		}
	}

	public synchronized void destroy() {
		BeanDescFactory.clear(beanDescBagMap.keySet());
		beanDescBagMap.clear();
		namedBeanDescList.clear();
		for (Iterator<ConfigBindResult> itr = bindMap.values().iterator(); itr
				.hasNext();) {
			ConfigBindResult result = itr.next();
			result.destroyBehaviors();
		}
		bindMap.clear();
	}

	@Override
	public void addPackageDesc(PackageDesc packageDesc) {
		Assertion.notNull(packageDesc);
		packageDescList.add(packageDesc);
	}

	@Override
	public PackageDesc getPackageDesc(String packageName) {
		Assertion.notNull(packageName);
		for (PackageDesc pd : packageDescList) {
			if (pd.getPackageName().equals(packageName)) {
				return pd;
			}
		}
		return null;
	}

	@Override
	public int getPackageDescSize() {
		return packageDescList.size();
	}

	@Override
	public List<PackageDesc> getPackageDescs() {
		return packageDescList;
	}

	@Override
	public List<Class<?>> getClassList(String packageName) {
		PackageDesc packageDesc = getPackageDesc(Assertion.notNull(packageName));
		if (packageDesc != null) {
			return packageDesc.getClassList();
		}
		return Collections.emptyList();
	}

	protected static <T> void assertComponentAnnotation(
			Class<? extends T> componentClass) {
		if (PackageUtil.isJavaPackage(componentClass)) {
			return;
		}
		if (!AnnotationUtil.hasAnnotationType(Component.class, componentClass
				.getAnnotations())) {
			throw new NoComponentAnnotationException(componentClass);
		}
	}

	@SuppressWarnings("unchecked")
	public <T> BeanDesc<T>[] getBeanDescs(Class<? super T> key) {
		return (BeanDesc<T>[]) getBeanDescList(key).toArray(
				(T[]) Array.newInstance(key, 0));
	}

	@SuppressWarnings("unchecked")
	public <T> T[] getAll(Class<? super T> key) {
		final List<BeanDesc<T>> beanDescList = getBeanDescList(key);
		T[] components = (T[]) Array.newInstance(key, beanDescList.size());
		int idx = 0;
		for (BeanDesc<T> beanDesc : beanDescList) {
			components[idx++] = buildComponent(key, beanDesc);
		}
		return components;
	}

	@SuppressWarnings("unchecked")
	protected <T> List<BeanDesc<T>> getBeanDescList(Class<? super T> key) {
		BeanDescBag beanDescBag = beanDescBagMap.get(key);
		if (beanDescBag != null) {
			return beanDescBag.getBeanDescList();
		} else {
			return Collections.emptyList();
		}
	}

	@SuppressWarnings("unchecked")
	protected <T> NameDesc<T> createNameDesc(Object identifier) {
		if (NameDesc.class.isInstance(identifier)) {
			return NameDesc.class.cast(identifier);
		}
		return new NameDescImpl<T>(identifier);
	}

	@Override
	public void load(Object componentDefinition) {
		Assertion.notNull(componentDefinition);
		LucyComponentRegistrar<Object> registrar = (LucyComponentRegistrar<Object>) lucyConf
				.getAcceptableComponentRegistrar(componentDefinition);
		if (registrar != null) {
			registrar.register(this, componentDefinition);
		} else {
			throw new LucyComponentRegistrarNotFoundRuntimeException(
					componentDefinition);
		}
	}

	@Override
	public <T> boolean hasComponent(Class<T> key) {
		return getBeanDesc(Assertion.notNull(key)) != null;
	}

	@Override
	public boolean hasComponent(Object identifier) {
		return getBeanDesc(Assertion.notNull(identifier)) != null;
	}

	@SuppressWarnings("unchecked")
	@Override
	public synchronized <T> T injectDependency(T t) {
		register(t);
		Class clazz = t.getClass();
		return (T) get(clazz);
	}

	protected static class BuildingPath {

		List<BeanDesc<?>> beanDescList = new ArrayList<BeanDesc<?>>();

		public void add(BeanDesc<?> beanDesc) {
			beanDescList.add(beanDesc);
		}

		public boolean isEmpty() {
			return beanDescList.isEmpty();
		}

		public void remove(BeanDesc<?> beanDesc) {
			beanDescList.remove(beanDescList.size() - 1);
		}

		public boolean isSingletonPresent() {
			for (BeanDesc<?> bd : beanDescList) {
				if (bd.isSingleton()) {
					return true;
				}
			}
			return false;
		}

		public boolean contains(BeanDesc<?> beanDesc) {
			for (BeanDesc<?> bd : beanDescList) {
				if (bd == beanDesc) {
					return true;
				}
			}
			return false;
		}
	}
}
