package lucy;

import java.lang.reflect.Constructor;
import java.util.Map;

import lucy.behavior.AspectBehavior;
import lucy.behavior.DestroyMethodBehavior;
import lucy.behavior.InitMethodBehavior;
import lucy.behavior.InjectBehavior;
import lucy.behavior.PrototypeBehavior;
import lucy.behavior.SingletonBehavior;
import lucy.spi.Behavior;
import lucy.spi.LucyConfiguration;

import commons.annotation.ConfigurationTarget;
import commons.exception.NoConfigurationTargetAnnotationException;
import commons.meta.BeanDesc;
import commons.meta.ClassDesc;
import commons.meta.ConfigDesc;
import commons.meta.ConfigDescContainer;
import commons.meta.ConstructorDesc;
import commons.meta.MethodDesc;
import commons.meta.PropertyDesc;
import commons.meta.impl.ConstructorDescImpl;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.Reflections.ConstructorUtil;

/**
 * ConfigDescBinderFactory is a factory class of ConfigDescBinder.
 * 
 * @author shot
 * 
 */
public class ConfigDescBinderFactory {

	private static ConfigDescBinder binder = new DefaultConfigDescBinderImpl();

	public static void setConfigDescBinder(ConfigDescBinder binder) {
		ConfigDescBinderFactory.binder = Assertion.notNull(binder);
	}

	public static ConfigDescBinder getConfigDescBinder() {
		return binder;
	}

	public static ConfigDescBinder getConfigDescBinder(LucyConfiguration conf) {
		Assertion.notNull(conf);
		return binder.loadConfiguration(conf);
	}

	public static void addBehavior(Class<? extends Behavior> b) {
		Assertion.notNull(b);
		binder.addBehavior(b);
	}

	public static class DefaultConfigDescBinderImpl implements ConfigDescBinder {

		protected Map<Class<?>, Class<? extends Behavior>> customBehaviorMap = CollectionsUtil
				.newHashMap();

		protected Map<Class<?>, Class<? extends Behavior>> defaultBehaviorMap = CollectionsUtil
				.newHashMap();

		protected Map<Class<? extends Behavior>, ConstructorDesc<Behavior>> behaviorConstructorCache = CollectionsUtil
				.newHashMap();

		protected LucyConfiguration conf;

		public DefaultConfigDescBinderImpl() {
			init();
		}

		protected void init() {
			addDefaultBehavior(InjectBehavior.class);
			addDefaultBehavior(SingletonBehavior.class);
			addDefaultBehavior(PrototypeBehavior.class);
			addDefaultBehavior(InitMethodBehavior.class);
			addDefaultBehavior(DestroyMethodBehavior.class);
			addDefaultBehavior(AspectBehavior.class);
		}

		protected void addDefaultBehavior(
				Class<? extends Behavior> behaviorClass) {
			addBehavior(behaviorClass, this.defaultBehaviorMap);
		}

		protected void addBehavior(Class<? extends Behavior> behaviorClass,
				Map<Class<?>, Class<? extends Behavior>> map) {
			ConfigurationTarget a = behaviorClass
					.getAnnotation(ConfigurationTarget.class);
			if (a == null) {
				throw new NoConfigurationTargetAnnotationException(
						behaviorClass);
			}
			for (Class<?> targetClass : a.value()) {
				map.put(targetClass, behaviorClass);
			}
		}

		@Override
		public void addBehavior(Class<? extends Behavior> behavior) {
			Assertion.notNull(behavior);
			addBehavior(behavior, this.customBehaviorMap);
		}

		@Override
		public synchronized <T> ConfigBindResult bind(BeanDesc<T> beanDesc) {
			Assertion.notNull(beanDesc);
			ConfigBindResult result = new ConfigBindResult(beanDesc);
			ClassDesc<T> cd = beanDesc.getClassDesc();
			for (int i = 0; i < cd.getConfigDescSize(); ++i) {
				ConfigDesc configDesc = cd.getConfigDesc(i);
				Behavior b = createBehavior(beanDesc, configDesc, cd);
				if (b != null) {
					result.addBehavior(cd, b);
				}
			}
			for (MethodDesc md : beanDesc.getAllMethodDesc()) {
				for (int i = 0; i < md.getConfigDescSize(); ++i) {
					ConfigDesc configDesc = md.getConfigDesc(i);
					Behavior b = createBehavior(beanDesc, configDesc, md);
					if (b != null) {
						result.addBehavior(md, b);
					}
				}
			}
			for (PropertyDesc<?> pd : beanDesc.getAllPropertyDesc()) {
				for (int i = 0; i < pd.getConfigDescSize(); ++i) {
					ConfigDesc configDesc = pd.getConfigDesc(i);
					Behavior b = createBehavior(beanDesc, configDesc, pd);
					if (b != null) {
						result.addBehavior(pd, b);
					}
				}
			}
			return result;
		}

		protected <T> Behavior createBehavior(BeanDesc<T> beanDesc,
				ConfigDesc configDesc, ConfigDescContainer container) {
			final Class<?> type = configDesc.getType();
			return createBehavior(beanDesc, type, configDesc, container);
		}

		/**
		 * Creating behavior using constructors.The construcors' parameters must
		 * be a BeanDesc and a ConfigDesc.
		 * 
		 * @param <T>
		 * @param beanDesc
		 * @param type
		 * @param configDesc
		 * @param container
		 * @return
		 */
		protected <T> Behavior createBehavior(BeanDesc<T> beanDesc,
				Class<?> type, ConfigDesc configDesc,
				ConfigDescContainer container) {
			final Class<? extends Behavior> clazz = findBehaviorClass(type);
			if (clazz != null) {
				ConstructorDesc<Behavior> cd = this.behaviorConstructorCache
						.get(clazz);
				if (cd == null) {
					cd = new ConstructorDescImpl<Behavior>(clazz);
					this.behaviorConstructorCache.put(clazz, cd);
				}
				Constructor<Behavior> c = cd.getSuitableConstructor(beanDesc,
						container, configDesc);
				return ConstructorUtil.newInstance(c, beanDesc, container,
						configDesc);
			} else {
				return null;
			}
		}

		protected Class<? extends Behavior> findBehaviorClass(Class<?> type) {
			Class<? extends Behavior> clazz = this.customBehaviorMap.get(type);
			if (clazz == null) {
				clazz = this.defaultBehaviorMap.get(type);
			}
			return clazz;
		}

		@Override
		public ConfigDescBinder loadConfiguration(final LucyConfiguration conf) {
			this.conf = Assertion.notNull(conf);
			return this;
		}

	}
}
