package lucy;

import java.util.List;

import lucy.spi.Behavior;
import lucy.spi.ComponentBuilder;
import lucy.spi.LucyConfiguration;

import commons.meta.BeanDesc;
import commons.meta.DescType;
import commons.meta.MethodDesc;
import commons.meta.PropertyDesc;
import commons.util.Assertion;

/**
 * ComponentBuilderFactory creates ComponentBuilder.Unlike it is,
 * ComponentBuilderFactory should be extensible for some reason, but not now.
 * 
 * @author shot
 */
public class ComponentBuilderFactory {

	public static ComponentBuilder createBuilder(BeanDesc<?> beanDesc, Lucy lucy) {
		return new DefaultComponentBuilderImpl(beanDesc, lucy);
	}

	public static ComponentBuilder createBuilder(BeanDesc<?> beanDesc,
			Lucy lucy, LucyConfiguration conf) {
		return createBuilder(beanDesc, lucy).loadConfiguration(conf);
	}

	public static class DefaultComponentBuilderImpl implements ComponentBuilder {

		protected BeanDesc<?> beanDesc;

		protected Lucy lucy;

		protected LucyConfiguration conf;

		public DefaultComponentBuilderImpl(BeanDesc<?> beanDesc, Lucy lucy) {
			this.beanDesc = beanDesc;
			this.lucy = lucy;
		}

		@Override
		public <T> T build(T t, ConfigBindResult result) {
			Assertion.notNull(result);
			for (Lifecycle current : Lifecycle.getComponentLifecycle()) {
				t = processClassDesc(t, result, current);
				processMethodAndProperties(t, result, current);
			}
			return t;
		}

		protected <T> T processClassDesc(T t, ConfigBindResult result,
				Lifecycle current) {
			return processBehavior(t, result, current, beanDesc.getClassDesc());
		}

		protected <T> T processBehavior(T t, ConfigBindResult result,
				Lifecycle current, DescType desc) {
			List<Behavior> behaviorList = result.getBehaviorList(desc);
			for (Behavior behavior : behaviorList) {
				if (behavior.isApplied(current)) {
					t = behavior.execute(t, lucy);
				}
			}
			return t;
		}

		protected <T> T processMethodAndProperties(T t,
				ConfigBindResult result, Lifecycle current) {
			for (MethodDesc md : beanDesc.getAllMethodDesc()) {
				if (!result.hasMethodDesc(md)) {
					continue;
				}
				t = processBehavior(t, result, current, md);
			}
			for (PropertyDesc<?> pd : beanDesc.getAllPropertyDesc()) {
				if (!result.hasPropertyDesc(pd)) {
					continue;
				}
				t = processBehavior(t, result, current, pd);
			}
			return t;
		}

		@Override
		public void setLucy(Lucy lucy) {
			this.lucy = lucy;
		}

		@Override
		public void dispose() {
			beanDesc = null;
			lucy = null;
		}

		@Override
		public ComponentBuilder loadConfiguration(LucyConfiguration conf) {
			this.conf = Assertion.notNull(conf);
			return this;
		}

	}
}
