package commons.util;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import commons.meta.BeanDesc;
import commons.meta.BeanDescFactory;
import commons.meta.PropertyDesc;
import commons.util.Reflections.ClassUtil;

/**
 * XML serialization utility.This class takes POJO object or simple instances
 * like String, Integer, and so on, then serializes these instances as xml.
 * 
 * @author shot
 * @since 0.2.0
 */
public class XMLSerializer {

	public static String HEADER = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>";

	public static List<Class<?>> simpleTypes = CollectionsUtil.newArrayList();

	static {
		simpleTypes.add(String.class);
		simpleTypes.add(Float.class);
		simpleTypes.add(Float.TYPE);
		simpleTypes.add(Double.class);
		simpleTypes.add(Double.TYPE);
		simpleTypes.add(Integer.class);
		simpleTypes.add(Integer.TYPE);
		simpleTypes.add(Boolean.class);
		simpleTypes.add(Boolean.TYPE);
		simpleTypes.add(Byte.class);
		simpleTypes.add(Byte.TYPE);
		simpleTypes.add(Short.class);
		simpleTypes.add(Short.TYPE);
		simpleTypes.add(Long.class);
		simpleTypes.add(Long.TYPE);
		simpleTypes.add(BigDecimal.class);
		simpleTypes.add(Date.class);
	}

	public static String serialize(Object target) {
		Assertion.notNull(target);
		StringBuilder builder = new StringBuilder(100);
		builder.append(HEADER);
		appendSerializeTarget(builder, target, true);
		return builder.toString();
	}

	private static String serialize(Object target, StringBuilder builder,
			boolean renderType) {
		Assertion.notNull(target);
		appendSerializeTarget(builder, target, renderType);
		return builder.toString();
	}

	private static void appendSerializeTarget(StringBuilder builder,
			Object target, boolean renderType) {
		if (target == null) {
			builder.append("null");
		} else if (String.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeString(builder, (String) target);
			} else {
				builder.append("<String>");
				appendSerializeString(builder, (String) target);
				builder.append("</String>");
			}
		} else if (Boolean.class.isInstance(target)
				|| Integer.class.isInstance(target)
				|| Byte.class.isInstance(target)
				|| Short.class.isInstance(target)
				|| Long.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeSimpleType(builder, target);
			} else {
				String name = ClassUtil.getShortClassName(target.getClass());
				builder.append("<");
				builder.append(name);
				builder.append(">");
				appendSerializeSimpleType(builder, target);
				builder.append("</");
				builder.append(name);
				builder.append(">");
			}
		} else if (BigDecimal.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeBigDecimal(builder, (BigDecimal) target);
			} else {
				builder.append("<BigDecimal>");
				appendSerializeBigDecimal(builder, (BigDecimal) target);
				builder.append("</BigDecimal>");
			}
		} else if (Date.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeDate(builder, (Date) target);
			} else {
				builder.append("<Date>");
				appendSerializeDate(builder, (Date) target);
				builder.append("</Date>");
			}
		} else if (Float.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeFloat(builder, (Float) target);
			} else {
				builder.append("<Float>");
				appendSerializeFloat(builder, (Float) target);
				builder.append("</Float>");
			}
		} else if (Double.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeDouble(builder, (Double) target);
			} else {
				builder.append("<Float>");
				appendSerializeDouble(builder, (Double) target);
				builder.append("</Float>");
			}
		} else if (List.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeList(builder, (List<?>) target, renderType);
			} else {
				builder.append("<List>");
				appendSerializeList(builder, (List<?>) target, renderType);
				builder.append("</List>");
			}
		} else if (Map.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeMap(builder, (Map<?, ?>) target, renderType);
			} else {
				builder.append("<Map>");
				appendSerializeMap(builder, (Map<?, ?>) target, renderType);
				builder.append("</Map>");
			}
		} else if (Enum.class.isInstance(target)) {
			if (!renderType) {
				appendSerializeString(builder, ((Enum<?>) target).name());
			} else {
				builder.append("<Enum>");
				appendSerializeString(builder, ((Enum<?>) target).name());
				builder.append("</Enum>");
			}
		} else if (target.getClass().isArray()) {
			if (!renderType) {
				appendSerializeArray(builder, target, renderType);
			} else {
				builder.append("<Array>");
				appendSerializeArray(builder, target, renderType);
				builder.append("</Array>");
			}
		} else {
			appendSerializeObject(builder, target);
		}
	}

	private static void appendSerializeBigDecimal(StringBuilder builder,
			BigDecimal target) {
		builder.append(target.toPlainString());
	}

	private static void appendSerializeObject(StringBuilder builder,
			Object target) {
		BeanDesc<Object> beanDesc = BeanDescFactory.getBeanDesc(target);
		String className = ClassUtil.getSimpleClassName(target.getClass());
		final int index = className.indexOf("$");
		if (index > 0) {
			className = className.substring(index + 1);
		}
		builder.append("<" + className + ">");
		for (PropertyDesc<Object> pd : beanDesc.getAllPropertyDesc()) {
			String name = pd.getPropertyName();
			Class<?> type = pd.getPropertyType();
			boolean isSimpleType = simpleTypes.contains(type)
					|| List.class.isAssignableFrom(type)
					|| Map.class.isAssignableFrom(type)
					|| Enum.class.isAssignableFrom(type);
			if (isSimpleType) {
				builder.append("<" + name + ">");
			}
			appendSerializeTarget(builder, pd.getValue(target), false);
			if (isSimpleType) {
				builder.append("</" + name + ">");
			}
		}
		builder.append("</" + className + ">");
	}

	private static void appendSerializeDate(StringBuilder builder, Date target) {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		builder.append(formatter.format(target));
	}

	private static void appendSerializeArray(StringBuilder builder,
			Object target, boolean renderType) {
		final int len = Array.getLength(target);
		for (int i = 0; i < len; ++i) {
			Object o = Array.get(target, i);
			serialize(o, builder, renderType);
		}
	}

	private static void appendSerializeMap(StringBuilder builder,
			Map<?, ?> target, boolean renderType) {
		for (Map.Entry<?, ?> entry : target.entrySet()) {
			if (renderType) {
				builder.append("<Entry>");
				builder.append("<Key>");
			}
			serialize(entry.getKey(), builder, renderType);
			if (renderType) {
				builder.append("</Key>");
				builder.append("<Value>");
			} else {
				builder.append("=");
			}
			serialize(entry.getValue(), builder, renderType);
			if (renderType) {
				builder.append("</Value>");
				builder.append("</Entry>");
			} else {
				builder.append(", ");
			}
		}
		if (!renderType) {
			builder.setLength(builder.length() - 2);
		}
	}

	private static void appendSerializeString(StringBuilder builder, String str) {
		builder.append(quote(str));
	}

	private static void appendSerializeList(StringBuilder builder,
			List<?> target, boolean renderType) {
		for (Object o : target) {
			serialize(o, builder, renderType);
			if (!renderType) {
				builder.append(", ");
			}
		}
		if (!renderType) {
			builder.setLength(builder.length() - 2);
		}
	}

	private static void appendSerializeSimpleType(StringBuilder builder,
			Object simpleObj) {
		builder.append(simpleObj.toString());
	}

	private static void appendSerializeFloat(StringBuilder builder, Float f) {
		if (f.isNaN() || f.isInfinite()) {
			throw new IllegalArgumentException(f.toString());
		}
		appendSerializeSimpleType(builder, f);
	}

	private static void appendSerializeDouble(StringBuilder builder, Double d) {
		if (d.isNaN() || d.isInfinite()) {
			throw new IllegalArgumentException(d.toString());
		}
		appendSerializeSimpleType(builder, d);
	}

	private static String quote(String target) {
		char current = 0;
		int len = target.length();
		StringBuilder strbuilder = new StringBuilder();
		for (int i = 0; i < len; ++i) {
			current = target.charAt(i);
			switch (current) {
			case '\\':
			case '/':
			case '"':
				strbuilder.append('\\');
				strbuilder.append(current);
				break;
			case '\b':
				strbuilder.append("\\b");
				break;
			case '\t':
				strbuilder.append("\\t");
				break;
			case '\n':
				strbuilder.append("\\n");
				break;
			case '\f':
				strbuilder.append("\\f");
				break;
			case '\r':
				strbuilder.append("\\r");
				break;
			default:
				if (current < ' ') {
					String t = "000" + Integer.toHexString(current);
					strbuilder.append("\\u" + t.substring(t.length() - 4));
				} else {
					strbuilder.append(current);
				}
			}
		}
		return strbuilder.toString();
	}
}
