package commons.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import commons.exception.IORuntimeException;

/**
 * URL utility class.
 * 
 * @author shot
 * 
 */
public class URLUtil {

	protected static final Map<String, String> CANONICAL_PROTOCOLS = new HashMap<String, String>();

	protected static final String VALID_SCHEME_CHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+.-";

	static {
		CANONICAL_PROTOCOLS.put("wsjar", "jar"); // for WebSphere
	}

	private URLUtil() {
	}

	public static URLConnection openConnection(URL url) {
		try {
			URLConnection connection = url.openConnection();
			connection.setUseCaches(false);
			return connection;
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static InputStream openStream(URL url) {
		try {
			URLConnection connection = url.openConnection();
			connection.setUseCaches(false);
			return connection.getInputStream();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static URL create(String spec) {
		try {
			return new URL(spec);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static URL create(URL context, String spec) {
		try {
			return new URL(context, spec);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static String encode(final String s, final String enc) {
		try {
			return URLEncoder.encode(s, enc);
		} catch (final UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	public static String decode(final String s, final String enc) {
		try {
			return URLDecoder.decode(s, enc);
		} catch (final UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	public static String toCanonicalProtocol(final String protocol) {
		final String canonicalProtocol = CANONICAL_PROTOCOLS.get(protocol);
		if (canonicalProtocol != null) {
			return canonicalProtocol;
		}
		return protocol;
	}

	public static boolean isAbsoluteUrl(String url) {
		if (url == null) {
			return false;
		}
		int colonPos;
		if ((colonPos = url.indexOf(":")) == -1) {
			return false;
		}
		for (int i = 0; i < colonPos; i++) {
			if (VALID_SCHEME_CHARS.indexOf(url.charAt(i)) == -1) {
				return false;
			}
		}
		return true;
	}

}
