package commons.util;

import java.io.File;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.jar.JarFile;
import java.util.regex.Pattern;

import commons.util.Traversal.ClassHandler;

/**
 * 
 * @author shot
 * 
 */
public class Traverser {

	protected String rootPackageName;

	protected Pattern[] classPatterns;

	protected Pattern[] ignorePatterns;

	protected Class<?> referenceClass;

	protected Map<String, Strategy> strategies = new HashMap<String, commons.util.Traverser.Strategy>();

	public Traverser(final String rootPackageName,
			final String[] classPatternStrings,
			final String[] ignorePatternStrings, final Class<?> referenceClass) {

		strategies.put("file", new FileSystemStrategy());
		strategies.put("jar", new JarFileStrategy());
		strategies.put("zip", new ZipFileStrategy());
		strategies.put("code-source", new CodeSourceFileStrategy());

		this.rootPackageName = rootPackageName;
		this.classPatterns = toPatterns(classPatternStrings);
		this.ignorePatterns = toPatterns(ignorePatternStrings);
		this.referenceClass = referenceClass;
	}

	protected Pattern[] toPatterns(final String[] patternStrings) {

		Pattern[] patterns = new Pattern[patternStrings.length];
		for (int i = 0; i < patternStrings.length; i++) {
			patterns[i] = Pattern.compile(patternStrings[i]);
		}
		return patterns;
	}

	public void addStrategy(final String protocol, final Strategy strategy) {
		strategies.put(protocol, strategy);
	}

	public void clearStrategies() {
		strategies.clear();
	}

	public Strategy getStrategy(final String protocol) {
		return strategies.get(URLUtil.toCanonicalProtocol(protocol));
	}

	public void traverse(ClassHandler classHandler) {
		final String baseClassPath = ResourceUtil
				.getResourcePath(referenceClass);
		final URL url = ResourceUtil.getResource(baseClassPath);
		classHandler = wrap(classHandler);
		final Strategy strategy = getStrategy(url.getProtocol());
		strategy.handle(referenceClass, url, rootPackageName, classHandler);
	}

	protected ClassHandler wrap(final ClassHandler classHandler) {

		if (classHandler == null) {
			return null;
		}
		return new RegexClassHandler(classHandler);
	}

	protected class RegexClassHandler implements ClassHandler {

		protected ClassHandler handler;

		public RegexClassHandler(final ClassHandler handler) {

			this.handler = handler;
		}

		@Override
		public void processClass(final String packageName,
				final String shortClassName) {

			if (!containsPackage(packageName)) {
				return;
			}
			String className;
			if (packageName.length() == rootPackageName.length()) {
				className = shortClassName;
			} else {
				className = packageName.substring(rootPackageName.length() + 1)
						+ shortClassName;
			}
			if (!containsClass(className)) {
				return;
			}

			handler.processClass(packageName, shortClassName);
		}

		protected boolean containsClass(String className) {

			for (Pattern pattern : ignorePatterns) {
				if (pattern.matcher(className).matches()) {
					return false;
				}
			}
			for (Pattern pattern : classPatterns) {
				if (pattern.matcher(className).matches()) {
					return true;
				}
			}
			return false;
		}

		protected boolean containsPackage(String packageName) {

			return (packageName.equals(rootPackageName) || packageName
					.startsWith(rootPackageName + "."));
		}
	}

	public static interface Strategy {
		void handle(Class<?> referenceClass, URL url, String rootPackageName,
				ClassHandler handler);
	}

	protected static class FileSystemStrategy implements Strategy {

		public void handle(final Class<?> referenceClass, final URL url,
				final String rootPackageName, final ClassHandler handler) {
			final File rootDir = getRootDir(referenceClass, url);
			Traversal.traverse(rootDir, rootPackageName, handler);
		}

		protected File getRootDir(final Class<?> referenceClass, final URL url) {
			final String[] names = referenceClass.getName().split("\\.");
			File path = ResourceUtil.getFile(url);
			for (int i = 0; i < names.length; ++i) {
				path = path.getParentFile();
			}
			return path;
		}
	}

	protected static class JarFileStrategy implements Strategy {

		public void handle(final Class<?> referenceClass, final URL url,
				final String rootPackageName, final ClassHandler handler) {
			final JarFile jarFile = createJarFile(url);
			Traversal.traverseFromJarFile(jarFile, handler);
		}

		protected JarFile createJarFile(final URL url) {
			return JarFileUtil.toJarFile(url);
		}
	}

	protected static class ZipFileStrategy implements Strategy {

		public void handle(final Class<?> referenceClass, final URL url,
				final String rootPackageName, final ClassHandler handler) {
			final JarFile jarFile = createJarFile(url);
			Traversal.traverseFromJarFile(jarFile, handler);
		}

		protected JarFile createJarFile(final URL url) {
			final String jarFileName = ZipFileUtil.toZipFilePath(url);
			return JarFileUtil.create(new File(jarFileName));
		}
	}

	protected static class CodeSourceFileStrategy implements Strategy {

		public void handle(final Class<?> referenceClass, final URL url,
				final String rootPackageName, final ClassHandler handler) {
			final JarFile jarFile = createJarFile(url);
			Traversal.traverseFromJarFile(jarFile, handler);
		}

		protected JarFile createJarFile(final URL url) {
			final URL jarUrl = URLUtil.create("jar:file:" + url.getPath());
			return JarFileUtil.toJarFile(jarUrl);
		}
	}
}
