package commons.util;

import static commons.Constants.CLASS_SUFFIX;
import static commons.Constants.WAR_SUFFIX;
import static commons.Constants.WEB_INF_CLASSES_PATH;

import java.io.File;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

public class Traversal {

	public interface ClassHandler {
		void processClass(String packageName, String shortClassName);
	}

	public static interface TraverseStrategy<R> {
		void doTraverse(final R resource, final String packageName,
				final ClassHandler handler);
	}

	protected Traversal() {
	}

	public static void traverse(File rootDir, String rootPackage,
			ClassHandler handler) {
		Assertion.notNull(rootDir);
		Assertion.notNull(handler);
		FileSystemTraverseStrategy strategy = new FileSystemTraverseStrategy();
		traverseFromFileSystem(rootDir, rootPackage, handler, strategy);
	}

	@SuppressWarnings("unchecked")
	public static void traverseFromFileSystem(File rootDir, String rootPackage,
			ClassHandler handler, TraverseStrategy strategy) {
		Assertion.notNull(rootDir);
		Assertion.notNull(handler);
		final File packageDir = getPackageDir(rootDir, rootPackage);
		if (packageDir.exists()) {
			strategy.doTraverse(packageDir, rootPackage, handler);
		}
	}

	public static void traverseFromJarFile(final JarFile jarFile,
			final ClassHandler handler) {
		Assertion.notNull(jarFile);
		Assertion.notNull(handler);
		JarFileTraverseStrategy strategy = new JarFileTraverseStrategy();
		traverseFromJarFile(jarFile, handler, strategy);
	}

	@SuppressWarnings("unchecked")
	public static void traverseFromJarFile(final JarFile jarFile,
			final ClassHandler handler, final TraverseStrategy strategy) {
		Assertion.notNull(jarFile);
		strategy.doTraverse(jarFile, null, handler);
	}

	public static class FileSystemTraverseStrategy implements
			TraverseStrategy<File> {

		@Override
		public void doTraverse(File rootDir, String packageName,
				ClassHandler handler) {
			traverseFileSystem(rootDir, packageName, handler, this);
		}

	}

	public static class JarFileTraverseStrategy implements
			TraverseStrategy<JarFile> {

		@Override
		public void doTraverse(final JarFile jarFile, final String notUse,
				final ClassHandler handler) {
			final boolean hasWarExtension = jarFile.getName().endsWith(
					WAR_SUFFIX);
			final Enumeration<JarEntry> e = jarFile.entries();
			while (e.hasMoreElements()) {
				final JarEntry entry = e.nextElement();
				final String entryName = entry.getName().replace('\\', '/');
				if (entryName.endsWith(CLASS_SUFFIX)) {
					final int startPos = (hasWarExtension && entryName
							.startsWith(WEB_INF_CLASSES_PATH)) ? WEB_INF_CLASSES_PATH
							.length()
							: 0;
					final String className = entryName.substring(startPos,
							entryName.length() - CLASS_SUFFIX.length())
							.replace('/', '.');
					final int pos = className.lastIndexOf('.');
					final String packageName = (pos == -1) ? null : className
							.substring(0, pos);
					final String shortClassName = (pos == -1) ? className
							: className.substring(pos + 1);
					handler.processClass(packageName, shortClassName);
				}
			}
		}

	}

	private static void traverseFileSystem(File packageDir, String packageName,
			ClassHandler handler, TraverseStrategy<File> strategy) {
		File[] files = packageDir.listFiles();
		for (int i = 0; i < files.length; i++) {
			final File file = files[i];
			final String filename = file.getName();
			if (file.isDirectory()) {
				final String concatName = concatName(packageName, filename);
				traverseFileSystem(file, concatName, handler, strategy);
			} else if (filename.endsWith(CLASS_SUFFIX)) {
				final String shortClassName = filename.substring(0, filename
						.lastIndexOf(CLASS_SUFFIX));
				handler.processClass(packageName, shortClassName);
			}
		}
	}

	private static File getPackageDir(final File rootDir,
			final String rootPackage) {
		File packageDir = rootDir;
		if (rootPackage != null) {
			final String[] names = rootPackage.split("\\.");
			for (int i = 0; i < names.length; i++) {
				packageDir = new File(packageDir, names[i]);
			}
		}
		return packageDir;
	}

	private static String concatName(String s1, String s2) {
		if (StringUtil.isEmpty(s1) && StringUtil.isEmpty(s2)) {
			return null;
		}
		if (!StringUtil.isEmpty(s1) && StringUtil.isEmpty(s2)) {
			return s1;
		}
		if (StringUtil.isEmpty(s1) && !StringUtil.isEmpty(s2)) {
			return s2;
		}
		return s1 + '.' + s2;
	}

}
