package commons.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import commons.exception.FileNotFoundRuntimeException;
import commons.exception.IORuntimeException;

/**
 * Utility class of stream.
 * 
 * @author shot
 * 
 */
public class StreamUtil {

	public static BufferedInputStream createBufferedFileInputStream(String path)
			throws FileNotFoundRuntimeException {
		try {
			return new BufferedInputStream(new FileInputStream(path));
		} catch (FileNotFoundException e) {
			throw new FileNotFoundRuntimeException(e, path);
		}
	}

	public static BufferedInputStream createBufferedFileInputStream(File file)
			throws FileNotFoundRuntimeException {
		try {
			return new BufferedInputStream(new FileInputStream(file));
		} catch (FileNotFoundException e) {
			throw new FileNotFoundRuntimeException(e, file.getPath());
		}
	}

	public static BufferedOutputStream createBufferedFileOutputStream(
			String path) throws FileNotFoundRuntimeException {
		try {
			return new BufferedOutputStream(new FileOutputStream(path));
		} catch (FileNotFoundException e) {
			throw new FileNotFoundRuntimeException(e, path);
		}
	}

	public static BufferedOutputStream createBufferedFileOutputStream(File file)
			throws FileNotFoundRuntimeException {
		try {
			return new BufferedOutputStream(new FileOutputStream(file));
		} catch (FileNotFoundException e) {
			throw new FileNotFoundRuntimeException(e, file.getPath());
		}
	}

	public static FileInputStream createFileInputStream(File file)
			throws IORuntimeException {
		try {
			return new FileInputStream(file);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static FileOutputStream createFileOutputStream(File file)
			throws IORuntimeException {
		try {
			return new FileOutputStream(file);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static final void copy(InputStream is, OutputStream os)
			throws IORuntimeException {
		byte[] buf = new byte[8192];
		try {
			int n = 0;
			while ((n = is.read(buf, 0, buf.length)) != -1) {
				os.write(buf, 0, n);
			}
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static int available(InputStream is) throws IORuntimeException {
		try {
			return is.available();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static void close(InputStream is) throws IORuntimeException {
		close0(is);
	}

	public static void close(OutputStream os) throws IORuntimeException {
		close0(os);
	}

	public static void flush(OutputStream out) throws IORuntimeException {
		if (out == null) {
			return;
		}
		try {
			out.flush();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static byte[] getBytes(final InputStream is)
			throws IORuntimeException {
		Assertion.notNull(is);
		byte[] bytes = null;
		byte[] buf = new byte[8192];
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			int n = 0;
			while ((n = is.read(buf, 0, buf.length)) != -1) {
				baos.write(buf, 0, n);
			}
			bytes = baos.toByteArray();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally {
			close(is);
			close(baos);
		}
		return bytes;
	}

	private static void close0(Closeable c) {
		if (c == null) {
			return;
		}
		try {
			c.close();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

}
