package commons.util;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

import commons.exception.ClassNotFoundRuntimeException;
import commons.exception.IllegalAccessRuntimeException;
import commons.exception.InstantiationRuntimeException;
import commons.exception.InvocationTargetRuntimeException;
import commons.exception.NoSuchConstructorRuntimeException;
import commons.exception.NoSuchFieldRuntimeException;
import commons.exception.NoSuchMethodRuntimeException;

/**
 * Reflections is an utility class group to do some reflection. There are :
 * 
 * <pre>
 * -ClassLoaderUtil - PackageUtil - ClassUtil - ConstructorUtil - FieldUtil
 * 		- MethodUtil - GenericsUtil
 * </pre>
 * 
 * @author shot
 */
public class Reflections {

	public static class ClassLoaderUtil {

		public static ClassLoader getClassLoader(final Class<?> targetClass) {
			final ClassLoader contextClassLoader = Thread.currentThread()
					.getContextClassLoader();
			if (contextClassLoader != null) {
				return contextClassLoader;
			}
			final ClassLoader targetClassLoader = targetClass.getClassLoader();
			final ClassLoader thisClassLoader = ClassLoaderUtil.class
					.getClassLoader();
			if (targetClassLoader != null && thisClassLoader != null) {
				if (isAncestor(thisClassLoader, targetClassLoader)) {
					return thisClassLoader;
				}
				return targetClassLoader;
			}
			if (targetClassLoader != null) {
				return targetClassLoader;
			}
			if (thisClassLoader != null) {
				return thisClassLoader;
			}

			final ClassLoader systemClassLoader = ClassLoader
					.getSystemClassLoader();
			if (systemClassLoader != null) {
				return systemClassLoader;
			}
			throw new IllegalStateException();
		}

		protected static boolean isAncestor(ClassLoader cl,
				final ClassLoader other) {
			while (cl != null) {
				if (cl == other) {
					return true;
				}
				cl = cl.getParent();
			}
			return false;
		}
	}

	public static class PackageUtil {

		private PackageUtil() {
		}

		public static String getPackageName(final Class<?> c) {
			Assertion.notNull(c);
			final Package p = c.getPackage();
			if (p == null) {
				final String name = c.getName();
				return name.substring(0, name.lastIndexOf("."));
			}
			return p.getName();
		}

		public static boolean isJavaPackage(final Class<?> c) {
			Assertion.notNull(c);
			String name = c.getName();
			if (name.startsWith("java.") || name.startsWith("javax.")) {
				return true;
			} else {
				return false;
			}
		}
	}

	public static class ClassUtil {

		private static Map<Class<?>, Class<?>> wrapperToPrimitiveMap = CollectionsUtil
				.newHashMap();

		private static Map<Class<?>, Class<?>> primitiveToWrapperMap = CollectionsUtil
				.newHashMap();

		static {
			wrapperToPrimitiveMap.put(Character.class, Character.TYPE);
			wrapperToPrimitiveMap.put(Byte.class, Byte.TYPE);
			wrapperToPrimitiveMap.put(Short.class, Short.TYPE);
			wrapperToPrimitiveMap.put(Integer.class, Integer.TYPE);
			wrapperToPrimitiveMap.put(Long.class, Long.TYPE);
			wrapperToPrimitiveMap.put(Double.class, Double.TYPE);
			wrapperToPrimitiveMap.put(Float.class, Float.TYPE);
			wrapperToPrimitiveMap.put(Boolean.class, Boolean.TYPE);

			primitiveToWrapperMap.put(Character.TYPE, Character.class);
			primitiveToWrapperMap.put(Byte.TYPE, Byte.class);
			primitiveToWrapperMap.put(Short.TYPE, Short.class);
			primitiveToWrapperMap.put(Integer.TYPE, Integer.class);
			primitiveToWrapperMap.put(Long.TYPE, Long.class);
			primitiveToWrapperMap.put(Double.TYPE, Double.class);
			primitiveToWrapperMap.put(Float.TYPE, Float.class);
			primitiveToWrapperMap.put(Boolean.TYPE, Boolean.class);
		}

		public static <T> Class<T> forName(final String className) {
			final String name = Assertion.notNull(className);
			final ClassLoader loader = Assertion.notNull(Thread.currentThread()
					.getContextClassLoader());
			return forName(name, loader);
		}

		@SuppressWarnings("unchecked")
		public static <T> Class<T> forName(final String className,
				final ClassLoader loader) {
			try {
				return (Class<T>) Class.forName(className, true, loader);
			} catch (final ClassNotFoundException e) {
				throw new ClassNotFoundRuntimeException(className, e);
			}
		}

		public static <T> Class<T> forNameNoException(final String className) {
			final String name = Assertion.notNull(className);
			final ClassLoader loader = Assertion.notNull(Thread.currentThread()
					.getContextClassLoader());
			return forNameNoException(name, loader);
		}

		@SuppressWarnings("unchecked")
		public static <T> Class<T> forNameNoException(final String className,
				final ClassLoader loader) {
			try {
				return (Class<T>) Class.forName(className, true, loader);
			} catch (final Throwable ignore) {
				return null;
			}
		}

		public static <T> T newInstanceByName(final String className) {
			Assertion.notNull(className);
			Class<T> c = forName(className);
			return (T) newInstance(c);
		}

		public static <T> T newInstanceByNameNoException(final String className) {
			Class<T> clazz = forNameNoException(className);
			try {
				return clazz.newInstance();
			} catch (final Throwable ignore) {
				return null;
			}
		}

		public static <T> T newInstance(final Class<T> clazz)
				throws InstantiationRuntimeException,
				IllegalAccessRuntimeException {
			try {
				return clazz.newInstance();
			} catch (final InstantiationException e) {
				throw new InstantiationRuntimeException(clazz, e);
			} catch (final IllegalAccessException e) {
				throw new IllegalAccessRuntimeException(clazz, e);
			}
		}

		@SuppressWarnings("unchecked")
		public static boolean isAssignableFrom(Class from, Class to) {
			final boolean fromPrimitive = from.isPrimitive();
			if (to == Object.class && !fromPrimitive) {
				return true;
			}
			final boolean toPrimitive = to.isPrimitive();
			if (!fromPrimitive && toPrimitive || fromPrimitive && !toPrimitive) {
				return false;
			}
			return to.isAssignableFrom(from);
		}

		public static Class<?> toWrapperClass(Class<?> clazz) {
			Class<?> c = primitiveToWrapperMap.get(clazz);
			if (c != null) {
				return c;
			} else {
				return clazz;
			}
		}

		public static String getClassName(Class<?> clazz) {
			Assertion.notNull(clazz);
			if (clazz.isArray()) {
				return clazz.getName() + "[]";
			} else {
				return clazz.getName();
			}
		}

		public static String getSimpleClassName(Class<?> clazz) {
			if (clazz == null) {
				return null;
			}
			String className = getClassName(clazz);
			return className.substring(className.lastIndexOf('.') + 1,
					className.length());
		}

		public static String getShortClassName(Class<?> clazz) {
			return getShortClassName(clazz.getCanonicalName());
		}

		public static String getShortClassName(String className) {
			int i = className.lastIndexOf('.');
			if (i > 0) {
				return className.substring(i + 1);
			}
			return className;
		}

	}

	public static class ConstructorUtil {

		public static <T> T newInstance(Constructor<T> constructor,
				Object... args) throws InstantiationRuntimeException,
				IllegalAccessRuntimeException, InvocationTargetRuntimeException {
			try {
				return constructor.newInstance(args);
			} catch (InstantiationException ex) {
				throw new InstantiationRuntimeException(constructor
						.getDeclaringClass(), ex);
			} catch (IllegalAccessException ex) {
				throw new IllegalAccessRuntimeException(constructor
						.getDeclaringClass(), ex);
			} catch (InvocationTargetException ex) {
				throw new InvocationTargetRuntimeException(constructor, ex);
			}
		}

		public static <T> boolean hasConstructor(Class<T> clazz,
				Class<?>... parameterTypes) {
			try {
				Constructor<T> c = clazz.getConstructor(parameterTypes);
				return c != null;
			} catch (SecurityException e) {
				return false;
			} catch (NoSuchMethodException e) {
				return false;
			}
		}

		public static <T> Constructor<T> getConstructor(Class<T> clazz,
				Class<?>... argTypes) throws NoSuchConstructorRuntimeException {
			try {
				return clazz.getConstructor(argTypes);
			} catch (NoSuchMethodException ex) {
				throw new NoSuchConstructorRuntimeException(clazz, argTypes, ex);
			}
		}

		@SuppressWarnings("unchecked")
		public static <T> Constructor<? extends T> findConstructorByParams(
				Class<? extends T> c, Object... args) {
			Assertion.notNull(c);
			Assertion.notNull(args);
			outer: for (Constructor<?> ct : c.getConstructors()) {
				Class<?>[] paramTypes = ct.getParameterTypes();
				if (paramTypes.length == args.length) {
					for (int j = 0; j < paramTypes.length; ++j) {
						if (!paramTypes[j].isAssignableFrom(args[j].getClass())) {
							continue outer;
						}
					}
					return (Constructor<? extends T>) ct;
				}
			}
			return null;
		}

	}

	public static class FieldUtil {

		public static Object get(Field field, Object target)
				throws IllegalAccessRuntimeException {

			try {
				return field.get(target);
			} catch (IllegalAccessException ex) {
				throw new IllegalAccessRuntimeException(field
						.getDeclaringClass(), field.getType(), ex);
			}
		}

		public static int getInt(Field field, Object target)
				throws IllegalAccessRuntimeException {

			try {
				return field.getInt(target);
			} catch (IllegalAccessException ex) {
				throw new IllegalAccessRuntimeException(field
						.getDeclaringClass(), field.getType(), ex);
			}

		}

		public static Field getField(final Class<?> clazz, final String name) {
			Assertion.notNulls(clazz, name);
			try {
				return clazz.getDeclaredField(name);
			} catch (SecurityException e) {
				throw e;
			} catch (NoSuchFieldException e) {
				throw new NoSuchFieldRuntimeException(e, clazz, name);
			}
		}

		public static Field getFieldWithAccessible(final Class<?> clazz,
				final String name) {
			Assertion.notNulls(clazz, name);
			try {
				final Field f = clazz.getDeclaredField(name);
				ReflectionExecutor
						.doAction(new ReflectionExecutor.ReflectionAction<Object>() {

							@Override
							public Object run() throws Exception {
								if (!f.isAccessible()) {
									f.setAccessible(true);
								}
								return null;
							}

						});
				return f;
			} catch (SecurityException e) {
				throw e;
			} catch (NoSuchFieldException e) {
				throw new NoSuchFieldRuntimeException(e, clazz, name);
			}
		}

		public static Field getFieldWithAccessibleNoException(
				final Class<?> clazz, final String name) {
			try {
				Field f = clazz.getDeclaredField(name);
				f.setAccessible(true);
				return f;
			} catch (Exception e) {
				return null;
			}
		}

		public static <T> Field getFieldWithAccessibleNoException(
				final List<Class<?>> list, final String name) {
			for (Class<?> c : list) {
				Field f = getFieldWithAccessibleNoException(c, name);
				if (f != null) {
					return f;
				}
			}
			return null;
		}

		public static void set(Field field, Object target, Object value)
				throws IllegalAccessRuntimeException {

			try {
				field.set(target, value);
			} catch (IllegalAccessException ex) {
				throw new IllegalAccessRuntimeException(field
						.getDeclaringClass(), field.getType(), ex);
			}

		}
	}

	public static class MethodUtil {

		public static Method getDeclaredMethod(Class<?> clazz,
				String methodName, Class<?>... parameterTypes) {
			Assertion.notNulls(clazz, methodName);
			try {
				return clazz.getDeclaredMethod(methodName, parameterTypes);
			} catch (SecurityException e) {
				throw e;
			} catch (NoSuchMethodException e) {
				throw new NoSuchMethodRuntimeException(clazz, methodName,
						parameterTypes, e);
			}
		}

		public static Method getDeclaredMethodNoException(Class<?> clazz,
				String methodName, Class<?>[] parameterTypes) {
			Assertion.notNulls(clazz, methodName);
			try {
				return clazz.getDeclaredMethod(methodName, parameterTypes);
			} catch (SecurityException e) {
				return null;
			} catch (NoSuchMethodException e) {
				return null;
			}
		}

		public static Method getMethodNoException(Class<?> clazz,
				String methodName, Class<?>[] parameterTypes) {
			Assertion.notNulls(clazz, methodName);
			try {
				return clazz.getMethod(methodName, parameterTypes);
			} catch (SecurityException e) {
				return null;
			} catch (NoSuchMethodException e) {
				return null;
			}
		}

		public static Method getMethod(Class<?> clazz, String methodName,
				Class<?>... argTypes) {
			try {
				return clazz.getMethod(methodName, argTypes);
			} catch (NoSuchMethodException ex) {
				throw new NoSuchMethodRuntimeException(clazz, methodName,
						argTypes, ex);
			}
		}

		@SuppressWarnings("unchecked")
		public static <T> T invoke(Method method, Object target, Object... args)
				throws InstantiationRuntimeException,
				IllegalAccessRuntimeException {
			try {
				return (T) method.invoke(target, args);
			} catch (InvocationTargetException ex) {
				Throwable t = ex.getCause();
				if (t instanceof RuntimeException) {
					throw (RuntimeException) t;
				}
				if (t instanceof Error) {
					throw (Error) t;
				}
				throw new InvocationTargetRuntimeException(method, ex);
			} catch (IllegalAccessException ex) {
				throw new IllegalAccessRuntimeException(method
						.getDeclaringClass(), method.getReturnType(), ex);
			}
		}

		public static boolean isAbstract(Method method) {
			int mod = method.getModifiers();
			return Modifier.isAbstract(mod);
		}

		public static String getSignature(String methodName,
				Class<?>... argTypes) {
			StringBuffer buf = new StringBuffer(100);
			buf.append(methodName);
			buf.append("(");
			if (argTypes != null) {
				for (int i = 0; i < argTypes.length; ++i) {
					if (i > 0) {
						buf.append(", ");
					}
					buf.append(argTypes[i].getName());
				}
			}
			buf.append(")");
			return buf.toString();
		}

		public static String getSignature(String methodName,
				Object... methodArgs) {
			StringBuffer buf = new StringBuffer(100);
			buf.append(methodName);
			buf.append("(");
			if (methodArgs != null) {
				for (int i = 0; i < methodArgs.length; ++i) {
					if (i > 0) {
						buf.append(", ");
					}
					if (methodArgs[i] != null) {
						buf.append(methodArgs[i].getClass().getName());
					} else {
						buf.append("null");
					}
				}
			}
			buf.append(")");
			return buf.toString();
		}

		public static boolean isEqualsMethod(Method method) {
			return method != null && method.getName().equals("equals")
					&& method.getReturnType() == boolean.class
					&& method.getParameterTypes().length == 1
					&& method.getParameterTypes()[0] == Object.class;
		}

		public static boolean isHashCodeMethod(Method method) {
			return method != null && method.getName().equals("hashCode")
					&& method.getReturnType() == int.class
					&& method.getParameterTypes().length == 0;
		}

		public static boolean isToStringMethod(Method method) {
			return method != null && method.getName().equals("toString")
					&& method.getReturnType() == String.class
					&& method.getParameterTypes().length == 0;
		}
	}

	public static class GenericsUtil {

		public static Class<?> getParameterizedClass(Method method) {
			final Type type = method.getGenericReturnType();
			if (isTypeOf(type, List.class)) {
				Type ret = getTypeOfList(type);
				if (ret != null) {
					return getRawClass(ret);
				}
			}
			return null;
		}

		public static Type getTypeOfList(final Type type) {
			if (!isTypeOf(type, List.class)) {
				return null;
			}
			return getGenericParameter(type, 0);
		}

		public static boolean isTypeOf(final Type type, final Class<?> clazz) {
			if (Class.class.isInstance(type)) {
				return clazz.isAssignableFrom(Class.class.cast(type));
			}
			if (ParameterizedType.class.isInstance(type)) {
				final ParameterizedType parameterizedType = ParameterizedType.class
						.cast(type);
				return isTypeOf(parameterizedType.getRawType(), clazz);
			}
			return false;
		}

		public static Type[] getGenericParameter(final Type type) {
			if (!ParameterizedType.class.isInstance(type)) {
				return null;
			}
			return ParameterizedType.class.cast(type).getActualTypeArguments();
		}

		public static Type getGenericParameter(final Type type, final int index) {
			if (!ParameterizedType.class.isInstance(type)) {
				return null;
			}
			final Type[] genericParameter = getGenericParameter(type);
			if (genericParameter == null) {
				return null;
			}
			return genericParameter[index];
		}

		public static Class<?> getRawClass(final Type type) {
			if (Class.class.isInstance(type)) {
				return Class.class.cast(type);
			}
			if (ParameterizedType.class.isInstance(type)) {
				final ParameterizedType parameterizedType = ParameterizedType.class
						.cast(type);
				return getRawClass(parameterizedType.getRawType());
			} else if (GenericArrayType.class.isInstance(type)) {
				GenericArrayType genericArrayType = GenericArrayType.class
						.cast(type);
				Class<?> c = getRawClass(genericArrayType
						.getGenericComponentType());
				return Array.newInstance(c, 0).getClass();
			}
			return null;
		}

	}
}
