package commons.util;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import commons.Disposable;
import commons.Disposer;

/**
 * Pattern utility.
 * 
 * @author shot
 * @see java.util.regex.Pattern
 * @see java.util.regex.Matcher
 */
public class PatternUtil {

	private static final Map<String, Pattern> patternCache = CollectionsUtil
			.newHashMap();

	protected PatternUtil() {
		Disposer.add(new Disposable() {

			@Override
			public void dispose() {
				clearPatternCache();
			}

		});
	}

	public static Pattern getPattern(String regex) {
		return getPattern(regex, 0);
	}

	public static Pattern getPattern(String regex, int flag) {
		Assertion.notNull(regex);
		Pattern pattern = (Pattern) patternCache.get(regex);
		if (pattern == null) {
			pattern = Pattern.compile(regex, flag);
			patternCache.put(regex, pattern);
		}
		return pattern;
	}

	public static boolean matches(String regex, CharSequence value) {
		Assertion.notNull(regex);
		Pattern pattern = getPattern(regex);
		Matcher matcher = pattern.matcher(value);
		return matcher.matches();
	}

	public static void clearPatternCache() {
		patternCache.clear();
	}

	public static List<Pattern> getPatterns(String[] regexes) {
		List<Pattern> patterns = CollectionsUtil.newArrayList();
		for (String regex : regexes) {
			if (regex == null) {
				continue;
			}
			patterns.add(getPattern(regex));
		}
		return patterns;
	}
}
