package commons.util;

import java.text.MessageFormat;
import java.util.ResourceBundle;

/**
 * Simple message formatter.
 * 
 * @author shot
 * 
 */
public class MessageFormatter {

	private static final String MESSAGES = "Messages";

	private MessageFormatter() {
	}

	public static String getMessage(String messageCode, Object[] args) {
		if (messageCode == null) {
			messageCode = "";
		}
		return "[" + messageCode + "]" + getSimpleMessage(messageCode, args);
	}

	/**
	 * get message from message resource without using any caches.
	 * 
	 * @param messageCode
	 * @param arguments
	 * @return
	 */
	public static String getSimpleMessageNoCache(String messageCode,
			Object[] arguments) {
		return getSimpleMessage0(messageCode, arguments, false);
	}

	/**
	 * get message from message resource.
	 * 
	 * @param messageCode
	 * @param arguments
	 * @return
	 */
	public static String getSimpleMessage(String messageCode, Object[] arguments) {
		return getSimpleMessage0(messageCode, arguments, true);
	}

	private static String getSimpleMessage0(String messageCode,
			Object[] arguments, boolean cacheable) {
		try {
			String pattern = getPattern(messageCode, cacheable);
			if (pattern != null) {
				return MessageFormat.format(pattern, arguments);
			}
		} catch (Throwable ignore) {
		}
		return getNoPatternMessage(arguments);
	}

	private static String getPattern(final String messageCode,
			final boolean cacheable) {
		ResourceBundle resourceBundle = getMessages(getSystemName(messageCode),
				cacheable);
		if (resourceBundle != null) {
			return resourceBundle.getString(messageCode);
		}
		return null;
	}

	private static String getSystemName(String messageCode) {
		return messageCode.substring(1, Math.min(4, messageCode.length()));
	}

	private static ResourceBundle getMessages(String systemName,
			boolean cacheable) {
		final String baseName = systemName + MESSAGES;
		if (cacheable) {
			return ResourceBundle.getBundle(baseName);
		} else {
			return ResourceBundleUtil.getBundleNoCache(baseName);
		}
	}

	private static String getNoPatternMessage(Object[] args) {
		if (args == null || args.length == 0) {
			return "";
		}
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < args.length; i++) {
			builder.append(args[i] + ", ");
		}
		builder.setLength(builder.length() - 2);
		return builder.toString();
	}
}