package commons.util;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import commons.meta.BeanDesc;
import commons.meta.BeanDescFactory;
import commons.meta.PropertyDesc;

/**
 * JSONSerializer is an utility class to convert between object to JSON.
 * 
 * @author shot
 */
public class JSONSerializer {

	private static final String COMMA = ",";

	private static final String COLON = ":";

	private static final String SINGLE_QUOTE = "'";

	private static final String DOUBLE_QUOTE = "\"";

	private static final String NULL_STRING = "null";

	private static final String TRUE_STRING = "true";

	private static final String FALSE_STRING = "false";

	private static final String START_BRACKET = "[";

	private static final String END_BRACKET = "]";

	private static final String START_BRACE = "{";

	private static final String END_BRACE = "}";

	public static <T> String serialize(T o) {
		StringBuilder builder = new StringBuilder(100);
		appendSerializeObject(builder, o);
		return builder.toString();
	}

	@SuppressWarnings("unchecked")
	public static void appendSerializeObject(StringBuilder builder, Object o) {
		if (o == null) {
			builder.append(JSONSerializer.NULL_STRING);
		} else if (o instanceof String) {
			builder.append(quote((String) o));
		} else if (Enum.class.isAssignableFrom(o.getClass())) {
			Enum e = (Enum) o;
			builder.append(quote(e.name()));
		} else if (o instanceof Float) {
			appendSerializeFloat(builder, (Float) o);
		} else if (o instanceof Double) {
			appendSerializeDouble(builder, (Double) o);
		} else if (o instanceof Number) {
			builder.append(o.toString());
		} else if (o instanceof Boolean) {
			appendSerializeBoolean(builder, (Boolean) o);
		} else if (o instanceof Collection) {
			appendSerializeArray(builder, ((Collection<?>) o).toArray());
		} else if (o instanceof Object[]) {
			appendSerializeArray(builder, (Object[]) o);
		} else if (o instanceof Map) {
			appendSerializeMap(builder, (Map<?, ?>) o);
		} else if (o.getClass().isArray()) {
			appendSerializeObjectArray(builder, o);
		} else {
			appendSerializeBean(builder, o);
		}
	}

	public static void appendSerializeFloat(StringBuilder builder, Float f) {
		if (f.isNaN() || f.isInfinite()) {
			throw new IllegalArgumentException(f.toString());
		}
		builder.append(f.toString());
	}

	public static void appendSerializeDouble(StringBuilder builder, Double d) {
		if (d.isNaN() || d.isInfinite()) {
			throw new IllegalArgumentException(d.toString());
		}
		builder.append(d.toString());
	}

	public static void appendSerializeBoolean(StringBuilder builder, Boolean b) {
		if (Boolean.TRUE.equals(b)) {
			builder.append(JSONSerializer.TRUE_STRING);
		} else {
			builder.append(JSONSerializer.FALSE_STRING);
		}
	}

	public static void appendSerializeArray(StringBuilder builder,
			Object[] array) {
		int length = array.length;
		builder.append(JSONSerializer.START_BRACKET);
		for (int i = 0; i < length; ++i) {
			appendSerializeObject(builder, array[i]);
			builder.append(JSONSerializer.COMMA);
		}
		if (length > 0) {
			builder.setLength(builder.length() - 1);
		}
		builder.append(JSONSerializer.END_BRACKET);
	}

	public static void appendSerializeMap(StringBuilder builder, Map<?, ?> map) {
		builder.append(JSONSerializer.START_BRACE);
		for (Iterator<?> i = map.keySet().iterator(); i.hasNext();) {
			String key = (String) i.next();
			builder.append(quote(key) + JSONSerializer.COLON);
			appendSerializeObject(builder, map.get(key));
			builder.append(JSONSerializer.COMMA);
		}
		if (map.size() > 0) {
			builder.setLength(builder.length() - 1);
		}
		builder.append(JSONSerializer.END_BRACE);
	}

	public static <T> void appendSerializeBean(StringBuilder builder, T t) {
		builder.append(JSONSerializer.START_BRACE);
		BeanDesc<T> beanDesc = BeanDescFactory.getBeanDesc(t);
		for (PropertyDesc<T> pd : beanDesc.getAllPropertyDesc()) {
			builder.append(pd.getPropertyName() + JSONSerializer.COLON);
			appendSerializeObject(builder, pd.getValue(t));
			builder.append(JSONSerializer.COMMA);
		}
		if (beanDesc.getPropertyDescSize() > 0) {
			builder.setLength(builder.length() - 1);
		}
		builder.append(JSONSerializer.END_BRACE);
	}

	public static void appendSerializeObjectArray(StringBuilder builder,
			Object o) {
		int length = Array.getLength(o);
		builder.append(JSONSerializer.START_BRACKET);
		for (int i = 0; i < length; i++) {
			Object value = Array.get(o, i);
			appendSerializeObject(builder, value);
			builder.append(JSONSerializer.COMMA);
		}
		if (length > 0) {
			builder.setLength(builder.length() - 1);
		}
		builder.append(JSONSerializer.END_BRACKET);
	}

	public static String quote(String str) {
		if (str == null || str.length() == 0) {
			return "\"\"";
		}
		char current = 0;
		int len = str.length();
		StringBuffer sb = new StringBuffer(len + 4);
		sb.append('"');
		for (int i = 0; i < len; ++i) {
			current = str.charAt(i);
			switch (current) {
			case '\\':
			case '/':
			case '"':
				sb.append('\\');
				sb.append(current);
				break;
			case '\b':
				sb.append("\\b");
				break;
			case '\t':
				sb.append("\\t");
				break;
			case '\n':
				sb.append("\\n");
				break;
			case '\f':
				sb.append("\\f");
				break;
			case '\r':
				sb.append("\\r");
				break;
			default:
				if (current < ' ') {
					String t = "000" + Integer.toHexString(current);
					sb.append("\\u" + t.substring(t.length() - 4));
				} else {
					sb.append(current);
				}
			}
		}
		sb.append('"');
		return sb.toString();
	}

	public static boolean isObject(String str) {
		return str.startsWith(JSONSerializer.START_BRACE)
				&& str.endsWith(JSONSerializer.END_BRACE);
	}

	public static boolean isArray(String str) {
		return str.startsWith(JSONSerializer.START_BRACKET)
				&& str.endsWith(JSONSerializer.END_BRACKET);
	}

	public static boolean isString(String str) {
		return (str.startsWith(JSONSerializer.SINGLE_QUOTE)
				&& str.endsWith(JSONSerializer.SINGLE_QUOTE) || str
				.startsWith(JSONSerializer.DOUBLE_QUOTE)
				&& str.endsWith(JSONSerializer.DOUBLE_QUOTE));
	}

	public static Object eval(String str) {
		if (JSONSerializer.isObject(str)) {
			return JSONSerializer.evalMap(str);
		} else if (JSONSerializer.isArray(str)) {
			return JSONSerializer.evalArray(str);
		} else if (JSONSerializer.isString(str)) {
			return JSONSerializer.evalString(str);
		} else {
			return JSONSerializer.evalInt(str);
		}
	}

	public static <T> Map<String, Object> evalMap(String str) {
		Map<String, Object> map = CollectionsUtil.newHashMap();
		StringBuffer buf = new StringBuffer();
		String key = null;
		String value = null;
		boolean valueParse = false;
		boolean isListedData = false;
		boolean isMappedData = false;
		for (int i = 0; i < str.length(); i++) {
			char c = str.charAt(i);
			switch (c) {
			case '{':
				if (valueParse) {
					String sub = str.substring(i, str.length() - 1);
					int count = 0;
					for (int j = 0; j < sub.length(); ++j) {
						char sc = sub.charAt(j);
						if (sc == '{') {
							count++;
							continue;
						}
						if (sc == '}' && count > 1) {
							count--;
						} else if (sc == '}' && count == 1) {
							String mapStr = sub.substring(0, j + 1);
							Map<String, Object> submap = JSONSerializer
									.evalMap(mapStr);
							map.put(key, submap);
							i = i + j;
							isMappedData = true;
							break;
						}
					}
				}
				break;
			case '}':
				break;
			case '[':
				if (valueParse) {
					String sub = str.substring(i, str.length() - 1);
					int count = 0;
					for (int j = 0; j < sub.length(); ++j) {
						char sc = sub.charAt(j);
						if (sc == '[') {
							count++;
							continue;
						}
						if (sc == ']' && count > 1) {
							count--;
						} else if (sc == ']' && count == 1) {
							String arrayStr = sub.substring(0, j + 1);
							List<Object> list = JSONSerializer
									.evalArray(arrayStr);
							map.put(key, list);
							i = i + j;
							isListedData = true;
							break;
						}
					}
				}
				break;
			case ']':
				break;
			case ':':
				key = JSONSerializer.evalString(buf.toString().trim());
				valueParse = true;
				buf = new StringBuffer();
				isListedData = false;
				isMappedData = false;
				break;
			case ',':
				if (!isListedData && !isMappedData) {
					value = buf.toString().trim();
					valueParse = false;
					buf = new StringBuffer();
					evalAndAddMap(key, value, map);
				}
				isListedData = false;
				isMappedData = false;
				break;
			default:
				buf.append(c);
				break;
			}
		}
		if (buf.length() > 0) {
			value = buf.toString().trim();
		}
		if (value != null) {
			evalAndAddMap(key, value, map);
		}
		return map;
	}

	public static List<Object> evalArray(String str) {
		List<Object> list = CollectionsUtil.newArrayList();
		StringBuffer buf = new StringBuffer();
		String value = null;
		boolean isMappedData = false;
		for (int i = 1; i < str.length() - 1; i++) {
			char c = str.charAt(i);
			switch (c) {
			case '{':
				String sub = str.substring(i, str.length() - 1);
				int count = 0;
				for (int j = 0; j < sub.length(); ++j) {
					char sc = sub.charAt(j);
					if (sc == '{') {
						count++;
						continue;
					}
					if (sc == '}' && count > 1) {
						count--;
					} else if (sc == '}' && count == 1) {
						String mapStr = sub.substring(0, j + 1);
						Map<String, Object> map = JSONSerializer
								.evalMap(mapStr);
						list.add(map);
						i = i + j;
						isMappedData = true;
						break;
					}
				}
				break;
			case '}':
				break;
			case '[':
				buf.append(c);
				break;
			case ']':
				buf.append(c);
				break;
			case ':':
				break;
			case ',':
				if (!isMappedData) {
					value = buf.toString().trim();
					buf = new StringBuffer();
					JSONSerializer.evalAndAddList(value, list);
				}
				isMappedData = false;
				break;
			default:
				buf.append(c);
				break;
			}
		}
		if (buf.length() > 0) {
			value = buf.toString().trim();
		}
		if (value != null) {
			JSONSerializer.evalAndAddList(value, list);
		}
		return list;
	}

	private static void evalAndAddList(String value, List<Object> list) {
		if (JSONSerializer.isObject(value)) {
			list.add(JSONSerializer.evalMap(value));
		} else if (JSONSerializer.isArray(value)) {
			list.add(JSONSerializer.evalArray(value));
		} else {
			Integer intValue = evalInt(value);
			if (intValue != null) {
				list.add(intValue);
			} else {
				list.add(JSONSerializer.evalString(value));
			}
		}
	}

	private static void evalAndAddMap(String key, String value,
			Map<String, Object> map) {
		if (JSONSerializer.isObject(value)) {
			Map<String, Object> v = JSONSerializer.evalMap(value);
			map.put(key, v);
		} else if (JSONSerializer.isArray(value)) {
			List<Object> list = JSONSerializer.evalArray(value);
			map.put(key, list);
		} else {
			Integer intValue = evalInt(value);
			if (intValue != null) {
				map.put(key, intValue);
			} else {
				map.put(key, JSONSerializer.evalString(value));
			}
		}
	}

	public static String evalString(String str) {
		if (JSONSerializer.isString(str)) {
			return str.substring(1, str.length() - 1);
		}
		return str;
	}

	public static Integer evalInt(String str) {
		try {
			int i = Integer.parseInt(str);
			return new Integer(i);
		} catch (NumberFormatException e) {
			return null;
		}
	}

}
