package commons.util;

import java.lang.annotation.Annotation;
import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.lang.reflect.AnnotatedElement;
import java.util.List;
import java.util.Set;

import commons.bootstrap.CommonsInitializer;
import commons.meta.AnnotationDesc;
import commons.meta.ConfigDesc;
import commons.meta.ConfigDescContainer;
import commons.meta.spi.AnnotationDescCreator;

/**
 * AnnotationUtil is an utility class for java.lang.annotation.Annotation. This
 * class depends on commons.bootstrap.CommonsInitializer,
 * commons.meta.spi.AnnotationDescCreator.
 * 
 * @author shot
 * @see commons.bootstrap.CommonsInitializer
 * @see commons.meta.spi.AnnotationDescCreator
 */
public class AnnotationUtil {

	protected static final Set<Class<? extends Annotation>> javaLangAnnotationSet = CollectionsUtil
			.newHashSet();

	protected static final AnnotationDescCreator creator;

	static {
		javaLangAnnotationSet.add(Retention.class);
		javaLangAnnotationSet.add(Target.class);
		javaLangAnnotationSet.add(Documented.class);
		javaLangAnnotationSet.add(Inherited.class);
		javaLangAnnotationSet.add(Deprecated.class);
		if (!CommonsInitializer.isInitialized()) {
			CommonsInitializer.init();
		}
		creator = CommonsInitializer.getFactory().getAnnotationDescCreator();
	}

	/**
	 * Return if given annotation is java.lang.annotation.*.
	 * 
	 * @param annotation
	 * @return
	 */
	public static boolean isJavaLangAnnotation(Annotation a) {
		Assertion.notNull(a);
		return javaLangAnnotationSet.contains(a.annotationType());
	}

	/**
	 * Create AnnotationDesc list from given annotatedElement.
	 * 
	 * @param container
	 * @param ae
	 * @param elementType
	 * @return
	 */
	public static List<ConfigDesc> createAnnotationDescList(
			final ConfigDescContainer container, final AnnotatedElement ae,
			final ElementType elementType) {
		Assertion.notNulls(ae, elementType);
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		for (Annotation a : ae.getDeclaredAnnotations()) {
			ConfigDesc[] configDescs = createAnnotationDescs(a);
			for (ConfigDesc ad : configDescs) {
				if (ad == null) {
					continue;
				}
				ad.setElementType(elementType);
				ad.setConfigDescContainer(container);
				list.add(ad);
			}
		}
		return list;
	}

	/**
	 * Return if given annotation is java.lang.annotation.Target.
	 * 
	 * @param a
	 * @return
	 */
	public static boolean isTargetAnnotation(Annotation a) {
		Assertion.notNull(a);
		return a.annotationType().equals(Target.class);
	}

	/**
	 * <pre>
	 * Get @Target from given annotation instance.
	 * </pre>
	 * 
	 * @param a
	 * @return
	 */
	public static Target getTargetAnnotation(Annotation a) {
		Assertion.notNull(a);
		return a.annotationType().getAnnotation(Target.class);
	}

	/**
	 * Return if annotation array has given annotation class. This method does
	 * recursive search up to top level annotation.
	 * 
	 * @param target
	 * @param annotations
	 * @return
	 */
	public static boolean hasAnnotationType(Class<? extends Annotation> target,
			Annotation[] annotations) {
		for (Annotation a : annotations) {
			if (target == a.annotationType()) {
				return true;
			}
		}
		for (Annotation a : annotations) {
			Annotation[] metas = a.annotationType().getAnnotations();
			if (metas == null || isTopLevelAnnotations(metas)) {
				continue;
			}
			boolean b = hasAnnotationType(target, metas);
			if (b) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Return if given annotations are all top level annotations or not.
	 * 
	 * @param annotations
	 * @return
	 */
	public static boolean isTopLevelAnnotations(Annotation[] annotations) {
		if (annotations == null) {
			return false;
		}
		for (Annotation a : annotations) {
			if (!isJavaLangAnnotation(a)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Create annotation desc arrays from an annotation instance.
	 * 
	 * @param a
	 * @return
	 */
	public static AnnotationDesc[] createAnnotationDescs(Annotation a) {
		return creator.createAnnotationDescs(a);
	}

	/**
	 * Create an annotation desc from an annotation instance.
	 * 
	 * @param a
	 * @return
	 */
	public static AnnotationDesc createAnnotationDesc(Annotation a) {
		return creator.createAnnotationDesc(a);
	}
}
