package commons.meta.spi;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Target;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Set;

import commons.Constants;
import commons.annotation.Composite;
import commons.annotation.Composite.EvaluateType;
import commons.meta.AnnotationDesc;
import commons.meta.LookupContext;
import commons.meta.impl.AnnotationDescImpl;
import commons.util.AnnotationUtil;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.Reflections.MethodUtil;

/**
 * Default implementation of AnnotationDescCreator.
 * 
 * @author shot
 * @see commons.meta.spi.AnnotationDescCreator
 */
public class DefaultAnnotationDescCreatorImpl implements AnnotationDescCreator {

	protected Class<? extends Annotation> compositeClass;

	protected Method compositeValueMethod;

	public DefaultAnnotationDescCreatorImpl() {
		this(Composite.class);
	}

	public DefaultAnnotationDescCreatorImpl(
			Class<? extends Annotation> compositeClass) {
		this.compositeClass = compositeClass;
		this.compositeValueMethod = MethodUtil.getDeclaredMethod(
				compositeClass, "value", Constants.EMPTY_CLASS_ARRAY);
	}

	@Override
	public AnnotationDesc[] createAnnotationDescs(final Annotation annotation) {
		Assertion.notNull(annotation);
		final boolean isComposite = isComposite(annotation);
		List<AnnotationDesc> retList = CollectionsUtil.newArrayList();
		Set<Class<? extends Annotation>> ignoreSet = CollectionsUtil
				.newHashSet();
		final AnnotationDesc ad = createAnnotationDesc(annotation);
		if (isComposite && shouldInclude(annotation)) {
			retList.add(ad);
		} else if (!isComposite) {
			retList.add(ad);
		}
		ignoreSet.add(annotation.annotationType());
		if (!isComposite) {
			return toArray(retList);
		}
		return toArray(walkAnnotations(retList, ad, ignoreSet));
	}

	protected boolean shouldInclude(Annotation a) {
		Annotation composite = a.annotationType().getAnnotation(compositeClass);
		EvaluateType type = MethodUtil.invoke(compositeValueMethod, composite,
				Constants.EMPTY_ARRAY);
		return type == EvaluateType.INCLUDE_THIS;
	}

	protected List<AnnotationDesc> walkAnnotations(List<AnnotationDesc> list,
			AnnotationDesc current, Set<Class<? extends Annotation>> ignoreSet) {
		for (Annotation meta : current.getMetaAnnotationList()) {
			final Class<? extends Annotation> metaType = meta.annotationType();
			if (!shouldIgnore(meta) || ignoreSet.contains(metaType)) {
				continue;
			}
			Target target = AnnotationUtil.getTargetAnnotation(meta);
			boolean accept = shouldAccept(target, current);
			if (accept) {
				final AnnotationDesc parent = createAnnotationDesc(meta);
				list.add(parent);
				if (isComposite(meta)) {
					ignoreSet.add(metaType);
				}
				if (current.hasLookup(metaType)) {
					final LookupContext context = current
							.getLookupContext(metaType);
					parent.setValues(context);
				}
				walkAnnotations(list, parent, ignoreSet);
			}
		}
		return list;
	}

	protected boolean shouldAccept(final Target target,
			final AnnotationDesc child) {
		if (target == null) {
			return true;
		} else {
			for (ElementType t : target.value()) {
				if (t == ElementType.ANNOTATION_TYPE) {
					continue;
				}
				if (child.isAcceptableType(t)) {
					return true;
				}
			}
		}
		return false;
	}

	@Override
	public AnnotationDesc createAnnotationDesc(final Annotation annotation) {
		return new AnnotationDescImpl(annotation);
	}

	protected boolean shouldIgnore(Annotation a) {
		if (a == null || a.annotationType() == compositeClass) {
			return false;
		}
		return true;
	}

	public boolean isComposite(Annotation a) {
		Assertion.notNull(a);
		return a.annotationType().getAnnotation(compositeClass) != null;
	}

	protected AnnotationDesc[] toArray(List<AnnotationDesc> list) {
		return list.toArray(new AnnotationDesc[list.size()]);
	}

}
