package commons.meta.impl;

import static commons.Constants.EMPTY_ARRAY;
import static commons.Constants.EMPTY_CLASS_ARRAY;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;

import commons.exception.IllegalPropertyRuntimeException;
import commons.exception.NoSuchGetterMethodRuntimeException;
import commons.exception.NoSuchSetterMethodRuntimeException;
import commons.meta.ClassDesc;
import commons.meta.MethodDesc;
import commons.meta.PropertyDesc;
import commons.util.Assertion;
import commons.util.AutoboxingUtil;
import commons.util.ConverterUtil;
import commons.util.JavaBeansUtil;
import commons.util.Reflections.FieldUtil;
import commons.util.Reflections.MethodUtil;

/**
 * Concrete class of PropertyDesc.
 * 
 * @author shot
 * 
 * @param <T>
 */
public class PropertyDescImpl<T> extends AbstractConfigDescContainer implements
		PropertyDesc<T> {

	protected MethodDesc readMethodDesc;

	protected MethodDesc writeMethodDesc;

	protected String propertyName;

	protected Class<?> propertyType;

	protected Class<T> targetClass;

	protected Field field;

	protected boolean array;

	protected Class<?> componentType;

	public PropertyDescImpl(ClassDesc<T> classDesc, Class<?> propertyType,
			String propertyName) {
		this(classDesc, propertyType, propertyName, false);
	}

	@SuppressWarnings("unchecked")
	public PropertyDescImpl(ClassDesc<T> classDesc, Class<?> propertyType,
			String propertyName, boolean findAuto) {
		this((Class<T>) classDesc.getConcreteClass(), propertyType,
				propertyName, findAuto);
	}

	public PropertyDescImpl(Class<T> targetClass, Class<?> propertyType,
			String propertyName) {
		this(targetClass, propertyType, propertyName, false);
	}

	public PropertyDescImpl(Class<T> targetClass, Class<?> propertyType,
			String propertyName, boolean findAuto) {
		this.propertyType = Assertion.notNull(propertyType);
		this.propertyName = Assertion.notNull(propertyName);
		this.targetClass = targetClass;
		this.array = propertyType.isArray();
		if (array) {
			this.componentType = propertyType.getComponentType();
		}
		init(findAuto);
	}

	protected void init(boolean findAuto) {
		if (targetClass == null) {
			return;
		}
		this.field = FieldUtil.getFieldWithAccessibleNoException(targetClass,
				propertyName);
		if (field == null) {
			return;
		}
		getConfigDescSupport().addAllAnnotationDesc(field);
		if (findAuto) {
			findReadWriteMethod(field);
		}
	}

	protected void findReadWriteMethod(Field field) {
		final String fieldName = field.getName();
		final Class<?> type = field.getType();
		String readMethodPrefix = "get";
		if (type == Boolean.TYPE || type == Boolean.class) {
			readMethodPrefix = "is";
		}
		final String readMethodName = readMethodPrefix
				+ JavaBeansUtil.capitalize(fieldName);
		final String writeMethodName = "set"
				+ JavaBeansUtil.capitalize(fieldName);
		try {
			Method rm = MethodUtil.getDeclaredMethod(getTargetClass(),
					readMethodName, EMPTY_CLASS_ARRAY);
			setReadMethod(rm);
		} catch (Exception ignore) {
		}
		try {
			Method wm = MethodUtil.getDeclaredMethod(getTargetClass(),
					writeMethodName, new Class[] { type });
			setWriteMethod(wm);
		} catch (Exception ignore) {
		}
	}

	public void setReadMethod(Method readMethod) {
		Assertion.notNull(readMethod);
		this.readMethodDesc = new MethodDescImpl(readMethod);
		support.addAllAnnotationDesc(readMethod);
	}

	public void setPropertyName(String propertyName) {
		this.propertyName = propertyName;
	}

	public void setWriteMethod(Method writeMethod) {
		Assertion.notNull(writeMethod);
		this.writeMethodDesc = new MethodDescImpl(writeMethod);
		support.addAllAnnotationDesc(writeMethod);
	}

	public void setPropertyType(Class<?> propertyType) {
		this.propertyType = propertyType;
	}

	public String getPropertyName() {
		return propertyName;
	}

	public Class<?> getPropertyType() {
		return propertyType;
	}

	public Method getReadMethod() {
		assertReadMethodDesc();
		return readMethodDesc.getMethod();
	}

	public Method getWriteMethod() {
		assertWriteMethodDesc();
		return writeMethodDesc.getMethod();
	}

	public Object getValue(T target) {
		assertReadMethodDesc();
		return readMethodDesc.invoke(target, EMPTY_ARRAY);
	}

	public void setValue(T target, Object... args) {
		assertWriteMethodDesc();
		try {
			final Object[] convertedArgs = convert(args);
			writeMethodDesc.invoke(target, convertedArgs);
		} catch (Throwable t) {
			throw new IllegalPropertyRuntimeException(t, targetClass,
					propertyName);
		}
	}

	protected Object[] convert(Object[] args) {
		if (args == null || Assertion.isAllNull(args)) {
			return convertNullToPrimitiveValueArgs(writeMethodDesc);
		}
		return convertArgs(args);
	}

	protected Object[] convertArgs(Object[] args) {
		Object[] ret = new Object[args.length];
		for (int i = 0; i < args.length; i++) {
			Object o = args[i];
			if (o == null) {
				continue;
			}
			Class<?> clazz = o.getClass();
			if (clazz.isArray()) {
				Class<?> type = clazz.getComponentType();
				ret[i] = convertArray(o, type.isPrimitive());
			} else {
				ret[i] = convert0(o);
			}
		}
		return ret;
	}

	protected Object convertArray(Object args, boolean primitive) {
		if (!isArray() || componentType == null) {
			throw new IllegalStateException(
					"convertArray must not work without componentType.");
		}
		final Object[] array = (!primitive) ? ((Object[]) args) : null;
		final int length = (!primitive) ? array.length : Array.getLength(args);
		Object ret = Array.newInstance(componentType, length);
		for (int i = 0; i < length; i++) {
			Object o = (!primitive) ? array[i] : Array.get(args, i);
			if (o == null) {
				continue;
			}
			Class<?> clazz = o.getClass();
			if (clazz.isArray()) {
				Class<?> type = clazz.getComponentType();
				Array.set(ret, i, convertArray(o, type.isPrimitive()));
			} else {
				Array.set(ret, i, convert0(o));
			}
		}
		return ret;
	}

	protected Object convert0(Object o) {
		Class<?> c;
		if (!isArray()) {
			c = propertyType;
		} else {
			c = componentType;
		}
		if (c.isPrimitive()) {
			return ConverterUtil.convert(o, c);
		} else if (Number.class.isAssignableFrom(c)) {
			return ConverterUtil.convert(o, c);
		} else if (Date.class.isAssignableFrom(c)) {
			return ConverterUtil.convertAsDate(o);
		} else if (Boolean.class == c) {
			return ConverterUtil.convertAsBoolean(o);
		} else if (Calendar.class.isAssignableFrom(c)) {
			return ConverterUtil.convertAsCalendar(o);
		} else if (String.class.isAssignableFrom(c)) {
			return ConverterUtil.convertAsString(o);
		}
		return o;
	}

	protected Object[] convertNullToPrimitiveValueArgs(final MethodDesc wmd) {
		final Class<?>[] paramTypes = wmd.getParameterTypes();
		final int len = paramTypes.length;
		Object[] ret = new Object[len];
		for (int i = 0; i < len; i++) {
			Class<?> c = paramTypes[i];
			if (c.isPrimitive()) {
				ret[i] = AutoboxingUtil.getDefaultPrimitiveValue(c);
			} else {
				ret[i] = null;
			}
		}
		return ret;
	}

	protected void assertWriteMethodDesc() {
		if (!isWritable()) {
			throw new NoSuchSetterMethodRuntimeException(targetClass,
					propertyName);
		}
	}

	protected void assertReadMethodDesc() {
		if (!isReadable()) {
			throw new NoSuchGetterMethodRuntimeException(targetClass,
					propertyName, propertyType);
		}
	}

	@Override
	public boolean isReadable() {
		return readMethodDesc != null;
	}

	@Override
	public boolean isWritable() {
		return writeMethodDesc != null;
	}

	@Override
	public MethodDesc getReadMethodDesc() {
		assertReadMethodDesc();
		return readMethodDesc;
	}

	@Override
	public MethodDesc getWriteMethodDesc() {
		assertWriteMethodDesc();
		return writeMethodDesc;
	}

	@Override
	public void setReadMethodDesc(MethodDesc readMethodDesc) {
		this.readMethodDesc = Assertion.notNull(readMethodDesc);
		support.addAllAnnotationDesc(readMethodDesc.getMethod());
	}

	@Override
	public void setWriteMethodDesc(MethodDesc writeMethodDesc) {
		this.writeMethodDesc = Assertion.notNull(writeMethodDesc);
		support.addAllAnnotationDesc(writeMethodDesc.getMethod());
	}

	@Override
	public Class<T> getTargetClass() {
		return targetClass;
	}

	@Override
	public boolean isArray() {
		return array;
	}

	public void setArray(boolean array) {
		this.array = array;
	}

}
