package commons.meta.impl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import commons.meta.AnnotationDesc;
import commons.meta.MethodDesc;
import commons.util.AnnotationUtil;
import commons.util.Assertion;
import commons.util.Reflections.MethodUtil;

/**
 * Concrete class of MethodDesc.
 * 
 * @author shot
 * 
 */
public class MethodDescImpl extends AbstractConfigDescContainer implements
		MethodDesc {

	protected Method method;

	protected String methodName;

	protected Class<?> returnType;

	protected Class<?>[] parameterTypes;

	protected int modifiers;

	protected Class<?>[] exceptionTypes;

	protected MethodParameter[] paramerters;

	protected AnnotationDesc[][] parameterAnnotationDesc;

	protected Annotation[][] parameterAnnotations;

	public MethodDescImpl(Method method) {
		this.method = Assertion.notNull(method);
		this.methodName = method.getName();
		this.returnType = method.getReturnType();
		this.parameterTypes = method.getParameterTypes();
		this.modifiers = method.getModifiers();
		this.exceptionTypes = method.getExceptionTypes();
		getConfigDescSupport().addAllAnnotationDesc(method);
		initParameters(method);
	}

	protected void initParameters(final Method method) {
		this.parameterAnnotations = method.getParameterAnnotations();
		if (parameterAnnotations.length == 0) {
			return;
		}
		this.paramerters = new MethodParameter[parameterAnnotations.length];
		for (int i = 0; i < parameterAnnotations.length; i++) {
			final Annotation[] annotations = parameterAnnotations[i];
			final MethodParameter param = new MethodParameter(i,
					parameterTypes[i], annotations);
			paramerters[i] = param;
		}
	}

	public Method getMethod() {
		return method;
	}

	public void setMethod(Method method) {
		this.method = method;
	}

	public String getMethodName() {
		return methodName;
	}

	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	public Class<?> getReturnType() {
		return returnType;
	}

	public void setReturnType(Class<?> returnType) {
		this.returnType = returnType;
	}

	public Class<?>[] getParameterTypes() {
		return parameterTypes;
	}

	public void setParameterTypes(Class<?>[] parameterTypes) {
		this.parameterTypes = parameterTypes;
	}

	@Override
	public <T> Object invoke(T t, Object... args) {
		final Method method = getMethod();
		return MethodUtil.invoke(method, t, args);
	}

	@Override
	public boolean isAbstract() {
		return Modifier.isAbstract(modifiers);
	}

	@Override
	public boolean isPublic() {
		return Modifier.isPublic(modifiers);
	}

	@Override
	public boolean isFinal() {
		return Modifier.isFinal(modifiers);
	}

	@Override
	public boolean isStatic() {
		return Modifier.isStatic(modifiers);
	}

	@Override
	public int getModifiers() {
		return modifiers;
	}

	@Override
	public Class<?>[] getExceptionTypes() {
		return exceptionTypes;
	}

	@Override
	public boolean equals(Object obj) {
		if (!MethodDesc.class.isInstance(obj)) {
			return false;
		}
		MethodDesc other = (MethodDesc) obj;
		if (this.getMethod().equals(other.getMethod())) {
			return true;
		}
		return false;
	}

	@Override
	public AnnotationDesc[][] getParameterAnnotationDescs() {
		if (parameterAnnotationDesc == null) {
			this.parameterAnnotationDesc = new AnnotationDesc[parameterTypes.length][];
			for (int i = 0; i < parameterAnnotationDesc.length; i++) {
				parameterAnnotationDesc[i] = getParameterAnnotationDescs(i);
			}
		}
		return parameterAnnotationDesc;
	}

	@Override
	public AnnotationDesc[] getParameterAnnotationDescs(int index) {
		return (paramerters != null && paramerters.length > index) ? paramerters[index]
				.getAnnotationDescs()
				: new AnnotationDesc[0];
	}

	@Override
	public int getParameterAnnotationDescSize() {
		return paramerters.length;
	}

	@Override
	public Annotation[][] getParameterAnnotations() {
		return parameterAnnotations;
	}

	public static class MethodParameter {

		final protected int index;

		final protected Class<?> type;

		final protected Annotation[] annotations;

		protected AnnotationDesc[] annotationDescs;

		public MethodParameter(int index, Class<?> type,
				Annotation[] annotations) {
			this.index = index;
			this.type = type;
			if (annotations != null) {
				this.annotations = annotations;
			} else {
				this.annotations = new Annotation[0];
			}
		}

		public AnnotationDesc[] getAnnotationDescs() {
			if (annotationDescs == null) {
				this.annotationDescs = createAnnotationDescs(annotations);
			}
			return annotationDescs;
		}

		protected AnnotationDesc[] createAnnotationDescs(
				Annotation[] annotations) {
			AnnotationDesc[] desc = new AnnotationDesc[annotations.length];
			int i = 0;
			for (Annotation a : annotations) {
				desc[i++] = AnnotationUtil.createAnnotationDesc(a);
			}
			return desc;
		}

		public int getIndex() {
			return index;
		}

		public Class<?> getType() {
			return type;
		}

		public Annotation[] getAnnotations() {
			return annotations;
		}

	}

}
