package commons.meta.impl;

import static commons.Constants.EMPTY_ARRAY;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import commons.meta.ClassDesc;
import commons.meta.ConstructorDesc;
import commons.util.Assertion;
import commons.util.Reflections.ClassUtil;

/**
 * Concrete class of ConstructorDesc.
 * 
 * @author shot
 * 
 * @param <T>
 */
public class ConstructorDescImpl<T> extends AbstractConfigDescContainer
		implements ConstructorDesc<T> {

	protected ClassDesc<T> classDesc = null;

	protected List<ConstructorInfo<T>> constructorInfoList = new ArrayList<ConstructorInfo<T>>();

	protected Constructor<?>[] constructors;

	protected Class<? extends T> targetClass;

	public ConstructorDescImpl(final ClassDesc<T> classDesc) {
		Assertion.notNull(classDesc);
		this.classDesc = classDesc;
		setup(classDesc);
	}

	public ConstructorDescImpl(final Class<? extends T> targetClass) {
		this.targetClass = Assertion.notNull(targetClass);
		setup(targetClass);
	}

	protected void setup(ClassDesc<T> classDesc) {
		final Class<? extends T> enhancedComponentClass = classDesc
				.getEnhancedComponentClass();
		this.targetClass = (enhancedComponentClass != null) ? enhancedComponentClass
				: classDesc.getComponentClass();
		setup(targetClass);
	}

	protected void setup(Class<? extends T> targetClass) {
		final Constructor<?>[] constructors = targetClass.getConstructors();
		this.constructors = analyzeConstructors(constructors);
		getConfigDescSupport().addAllAnnotationDesc(constructors);
	}

	@SuppressWarnings("unchecked")
	protected Constructor<?>[] analyzeConstructors(
			final Constructor<?>[] constructors) {
		Assertion.notNull(constructors);
		for (Constructor<?> c : constructors) {
			final Class<?>[] paramTypes = c.getParameterTypes();
			final int paramLength = paramTypes.length;
			final boolean[] primitives = new boolean[paramLength];
			boolean allNotPrimitives = true;
			for (int i = 0; i < primitives.length; i++) {
				final boolean prim = paramTypes[i].isPrimitive();
				primitives[i] = prim;
				if (prim) {
					allNotPrimitives = false;
				}
			}
			ConstructorInfo<T> info = new ConstructorInfo<T>(
					(Constructor<T>) c, paramTypes, paramLength, primitives,
					allNotPrimitives);
			constructorInfoList.add(info);
		}
		return constructors;
	}

	public final ClassDesc<T> getClassDesc() {
		return classDesc;
	}

	public Constructor<T> getSuitableConstructor(Object... args) {
		if (args == null) {
			args = EMPTY_ARRAY;
		}
		outer: for (int i = 0; i < constructorInfoList.size(); i++) {
			ConstructorInfo<T> info = constructorInfoList.get(i);
			if (info.getParamLength() != args.length) {
				continue;
			}
			for (int j = 0; j < args.length; j++) {
				if (args[j] == null) {
					continue;
				}
				final Class<? extends Object> argsClass = args[j].getClass();
				if (ClassUtil.isAssignableFrom(argsClass, info.getParamType(j))) {
					continue;
				}
				continue outer;
			}
			return info.getConstructor();
		}
		return null;
	}

	@Override
	public Constructor<?>[] getConstructors() {
		return constructors;
	}

	@Override
	public Constructor<T> getConstructor(Class<?>[] paramTypes) {
		for (ConstructorInfo<T> info : constructorInfoList) {
			Class<?>[] types = info.getParamTypes();
			if (Arrays.equals(paramTypes, types)) {
				return info.getConstructor();
			}
		}
		// TODO throw exception
		return null;
	}

	private static class ConstructorInfo<T> {

		private final Constructor<T> constructor;

		private final Class<?>[] paramTypes;

		private final int paramLength;

		private final boolean[] primitives;

		public ConstructorInfo(final Constructor<T> constructor,
				final Class<?>[] paramTypes, int paramLength,
				boolean[] primitives, boolean allNotPrimitives) {
			this.constructor = constructor;
			this.paramTypes = paramTypes;
			this.paramLength = paramLength;
			this.primitives = primitives;
		}

		public Constructor<T> getConstructor() {
			return constructor;
		}

		public int getParamLength() {
			return paramLength;
		}

		public Class<?>[] getParamTypes() {
			return paramTypes;
		}

		public boolean[] getPrimitives() {
			return primitives;
		}

		public Class<?> getParamType(int index) {
			if (index > getParamLength()) {
				return null;
			}
			return paramTypes[index];
		}

		public boolean isPrimitive(int index) {
			if (index > getParamLength()) {
				return false;
			}
			return primitives[index];
		}

	}
}
