package commons.meta.impl;

import java.io.Externalizable;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;

import commons.meta.ClassDesc;
import commons.util.Assertion;
import commons.util.CollectionsUtil;

/**
 * @author shot
 * 
 * @param <T>
 */
public class ClassDescImpl<T> extends AbstractConfigDescContainer implements
		ClassDesc<T> {

	protected String componentName;

	protected Class<? extends T> targetClass;

	protected Class<? extends T> enhancedTargetClass;

	protected List<Class<?>> dependencyClassesList = CollectionsUtil
			.newArrayList();

	protected Map<String, Class<?>[]> classesMap = CollectionsUtil.newHashMap();

	private static Set<Class<?>> unacceptableClasses = CollectionsUtil
			.newHashSet();

	static {
		init();
	}

	public ClassDescImpl(Class<? extends T> targetClass) {
		this(null, targetClass);
	}

	public ClassDescImpl(String componentName, Class<? extends T> targetClass) {
		this.targetClass = targetClass;
		setup(componentName);
	}

	protected void setup(String componentName) {
		analyzeClassDependency(targetClass);
		if (componentName != null) {
			this.componentName = componentName;
			setupComponentNameMap(componentName, dependencyClassesList);
		}
		getConfigDescSupport().addAllAnnotationDesc(targetClass);
	}

	protected void setupComponentNameMap(final String componentName,
			List<Class<?>> classesList) {
		final Class<?>[] array = classesList.toArray(new Class[classesList
				.size()]);
		classesMap.put(componentName, array);
	}

	protected void analyzeClassDependency(Class<? extends T> targetClass) {
		for (Class<?> c = targetClass; c != null && c != Object.class; c = c
				.getSuperclass()) {
			analyzeByInterface(c);
		}
	}

	protected void analyzeByInterface(final Class<?> c) {
		if (isUnacceptableClass(c)) {
			return;
		}
		addDependencyClass(c);
		final Class<?>[] interfaces = c.getInterfaces();
		for (Class<?> cl : interfaces) {
			analyzeByInterface(cl);
		}
	}

	protected void addDependencyClass(final Class<?> clazz) {
		dependencyClassesList.add(clazz);
	}

	public Map<String, Class<?>[]> getClassesMap() {
		return classesMap;
	}

	public Class<? extends T> getComponentClass() {
		return targetClass;
	}

	public List<Class<?>> getDependencyClassesList() {
		return dependencyClassesList;
	}

	public boolean isUnacceptableClass(Class<?> c) {
		return unacceptableClasses.contains(c);
	}

	public static void addUnacceptableClass(Class<?> c) {
		unacceptableClasses.add(Assertion.notNull(c));
	}

	public static void clearUnacceptableClass() {
		init();
	}

	public String getComponentName() {
		return componentName;
	}

	private static void init() {
		Assertion.notNull(unacceptableClasses).clear();
		unacceptableClasses.add(Cloneable.class);
		unacceptableClasses.add(Comparable.class);
		unacceptableClasses.add(Serializable.class);
		unacceptableClasses.add(Externalizable.class);
	}

	@Override
	public Class<? extends T> getEnhancedComponentClass() {
		return enhancedTargetClass;
	}

	@Override
	public void setEnhancedComponentClass(Class<? extends T> componentClass) {
		this.enhancedTargetClass = componentClass;
	}

	@Override
	public Class<? extends T> getConcreteClass() {
		return (enhancedTargetClass != null) ? enhancedTargetClass
				: targetClass;
	}

}
