package commons.meta.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import commons.meta.AspectConfigDesc;
import commons.meta.BeanDesc;
import commons.meta.BeanDescSupport;
import commons.meta.ClassDesc;
import commons.meta.ConfigDesc;
import commons.meta.ConfigDescContainer;
import commons.meta.ConstructorDesc;
import commons.meta.MethodDesc;
import commons.meta.NameDesc;
import commons.meta.PackageDesc;
import commons.meta.PackageDescHolder;
import commons.meta.PropertyDesc;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.JavaBeansUtil;
import commons.util.Reflections.ConstructorUtil;
import commons.util.Reflections.PackageUtil;

/**
 * Concrete class of BeanDesc.
 * 
 * @author shot
 * 
 * @param <T>
 */
public class BeanDescImpl<T> implements BeanDesc<T> {

	protected static final List<ConfigDesc> EMPTY_CONFIG_LIST = CollectionsUtil
			.emptyList();

	protected ClassDesc<T> classDesc;

	protected ConstructorDesc<T> constructorDesc;

	protected NameDesc<T> nameDesc;

	protected Map<ConfigDescContainer, List<ConfigDesc>> configMap;

	protected Map<String, PropertyDesc<T>> propertyDescMap;

	protected List<PropertyDesc<T>> propertyDescList;

	protected Map<String, List<MethodDesc>> methodDescMap;

	protected List<MethodDesc> methodDescList;

	protected T component;

	protected BeanDescSupport<T> support;

	protected PackageDesc packageDesc;

	protected List<AspectConfigDesc> aspectConfigDescList;

	protected boolean singleton;

	public BeanDescImpl(Class<? extends T> componentClass) {
		this(componentClass, null);
	}

	public BeanDescImpl(Class<? extends T> componentClass,
			PackageDescHolder holder) {
		Assertion.notNull(componentClass);
		init(componentClass, holder);
	}

	protected void init(Class<? extends T> componentClass,
			PackageDescHolder holder) {
		init0();
		analyzeClasses(componentClass);
		analyzeConstructors();
		analyzePropertiesAndMethods(componentClass);
		if (holder != null) {
			analyzePackages(holder);
		}
		support.storeAnnotationConfiguration(configMap);
	}

	protected void init0() {
		this.configMap = CollectionsUtil.newHashMap();
		this.propertyDescMap = CollectionsUtil.newHashMap();
		this.propertyDescList = CollectionsUtil.newArrayList();
		this.methodDescMap = CollectionsUtil.newHashMap();
		this.methodDescList = CollectionsUtil.newArrayList();
		this.support = new BeanDescSupport<T>(this);
		this.aspectConfigDescList = CollectionsUtil.newArrayList();
	}

	protected void analyzePackages(final PackageDescHolder holder) {
		final Class<? extends T> c = getClassDesc().getComponentClass();
		final String pkgname = PackageUtil.getPackageName(c);
		this.packageDesc = holder.getPackageDesc(pkgname);
		if (packageDesc == null) {
			packageDesc = new PackageDescImpl(this);
			holder.addPackageDesc(packageDesc);
		} else {
			packageDesc.addBeanDesc(this);
		}
	}

	protected void analyzeClasses(final Class<? extends T> componentClass) {
		setClassDesc(new ClassDescImpl<T>(componentClass));
	}

	protected void setClassDesc(ClassDesc<T> classDesc) {
		this.classDesc = classDesc;
	}

	protected void analyzePropertiesAndMethods(
			final Class<? extends T> componentClass) {
		for (Method m : componentClass.getMethods()) {
			final String methodName = m.getName();
			if (support.isIgnorableMethod(m)) {
				continue;
			}
			if (JavaBeansUtil.isGetMethod(methodName, m)) {
				String propertyName = JavaBeansUtil.decapitalize(methodName
						.substring("get".length()));
				setReadMethod(m, propertyName);
			} else if (JavaBeansUtil.isIsMethod(methodName, m)) {
				String propertyName = JavaBeansUtil.decapitalize(methodName
						.substring("is".length()));
				setReadMethod(m, propertyName);
			} else if (JavaBeansUtil.isSetMethod(methodName, m)) {
				String propertyName = JavaBeansUtil.decapitalize(methodName
						.substring("set".length()));
				setWriteMethod(m, propertyName);
			} else {
				final MethodDesc methodDesc = new MethodDescImpl(m);
				addMethodDesc(methodDesc);
			}
		}
	}

	protected void setReadMethod(Method readMethod, String propertyName) {
		PropertyDesc<T> propertyDesc = getPropertyDesc(propertyName);
		if (propertyDesc == null) {
			propertyDesc = new PropertyDescImpl<T>(getClassDesc(), readMethod
					.getReturnType(), propertyName);
			addPropertyDesc(propertyDesc);
		}
		propertyDesc.setPropertyName(propertyName);
		propertyDesc.setReadMethod(readMethod);
	}

	protected void setWriteMethod(Method writeMethod, String propertyName) {
		PropertyDesc<T> propertyDesc = getPropertyDesc(propertyName);
		if (propertyDesc == null) {
			final Class<?> propertyType = writeMethod.getParameterTypes()[0];
			propertyDesc = new PropertyDescImpl<T>(getClassDesc(),
					propertyType, propertyName);
			propertyDesc.setPropertyType(propertyType);
			addPropertyDesc(propertyDesc);
		}
		propertyDesc.setPropertyName(propertyName);
		propertyDesc.setWriteMethod(writeMethod);
	}

	protected void analyzeConstructors() {
		final ClassDesc<T> classDesc = getClassDesc();
		final ConstructorDesc<T> constructorDesc = new ConstructorDescImpl<T>(
				classDesc);
		setConstructorDesc(constructorDesc);
	}

	public List<ConfigDesc> getConfigDescList(ConfigDescContainer container) {
		Assertion.notNull(container);
		List<ConfigDesc> list = configMap.get(container);
		if (list != null) {
			return list;
		}
		return Collections.emptyList();
	}

	public ClassDesc<T> getClassDesc() {
		return classDesc;
	}

	public Class<? extends T> getComponentClass() {
		return classDesc.getComponentClass();
	}

	public MethodDesc getMethodDesc(String methodName) {
		final List<MethodDesc> list = methodDescMap.get(methodName);
		return (list != null && list.size() > 0) ? list.get(0) : null;
	}

	public List<MethodDesc> getMethodDescs(String methodName) {
		return methodDescMap.get(methodName);
	}

	public PropertyDesc<T> getPropertyDesc(String propertyName) {
		return propertyDescMap.get(propertyName);
	}

	public ConstructorDesc<T> getConstructorDesc() {
		return constructorDesc;
	}

	protected void setConstructorDesc(ConstructorDesc<T> constructorDesc) {
		this.constructorDesc = constructorDesc;
	}

	public T newInstance(Object... args) {
		final ConstructorDesc<T> cd = getConstructorDesc();
		final Constructor<T> c = cd.getSuitableConstructor(args);
		return ConstructorUtil.newInstance(c, args);
	}

	protected void addMethodDesc(MethodDesc desc) {
		final String methodName = desc.getMethodName();
		if (!methodDescMap.containsKey(methodName)) {
			List<MethodDesc> list = CollectionsUtil.newArrayList();
			list.add(desc);
			methodDescMap.put(methodName, list);
			methodDescList.add(desc);
		} else {
			final List<MethodDesc> list = methodDescMap.get(methodName);
			list.add(desc);
		}
	}

	protected void addPropertyDesc(PropertyDesc<T> pd) {
		propertyDescMap.put(pd.getPropertyName(), pd);
		propertyDescList.add(pd);
	}

	public List<MethodDesc> getAllMethodDesc() {
		return methodDescList;
	}

	public List<PropertyDesc<T>> getAllPropertyDesc() {
		return propertyDescList;
	}

	@Override
	public T getComponent() {
		return component;
	}

	public void setComponent(T component) {
		this.component = component;
	}

	@Override
	public NameDesc<T> getNameDesc() {
		return nameDesc;
	}

	@Override
	public void setNameDesc(NameDesc<T> nameDesc) {
		this.nameDesc = Assertion.notNull(nameDesc);
	}

	@Override
	public String getComponentName() {
		return (nameDesc != null) ? nameDesc.getIdentifier().toString() : null;
	}

	@Override
	public Class<? extends T> getEnhancedComponentClass() {
		return getClassDesc().getEnhancedComponentClass();
	}

	@Override
	public int getPropertyDescSize() {
		return propertyDescList.size();
	}

	@Override
	public void setEnhancedComponentClass(Class<? extends T> enhancedClass) {
		Assertion.notNull(enhancedClass);
		getClassDesc().setEnhancedComponentClass(enhancedClass);
		analyzeConstructors();
	}

	@Override
	public PackageDesc getPackageDesc() {
		return packageDesc;
	}

	protected BeanDescSupport<T> getBeanDescSupport() {
		return support;
	}

	protected Map<ConfigDescContainer, List<ConfigDesc>> getConfigMap() {
		return configMap;
	}

	@Override
	public boolean hasName() {
		return nameDesc != null;
	}

	@Override
	public List<ConfigDesc> getClassConfigDescList() {
		return configMap.containsKey(getClassDesc()) ? configMap
				.get(getClassDesc()) : EMPTY_CONFIG_LIST;
	}

	@Override
	public List<ConfigDesc> getConstructorConfigDescList() {
		return configMap.containsKey(getConstructorDesc()) ? configMap
				.get(getConstructorDesc()) : EMPTY_CONFIG_LIST;
	}

	@Override
	public List<ConfigDesc> getMethodConfigDescList() {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		for (MethodDesc md : getAllMethodDesc()) {
			addConfigDescTo(list, md);
		}
		for (PropertyDesc<T> pd : getAllPropertyDesc()) {
			addConfigDescTo(list, pd);
		}
		return list;
	}

	protected void addConfigDescTo(List<ConfigDesc> list,
			ConfigDescContainer container) {
		for (int i = 0; i < container.getConfigDescSize(); i++) {
			list.add(container.getConfigDesc(i));
		}
	}

	@Override
	public List<ConfigDesc> getPackageConfigDescList() {
		return configMap.containsKey(getPackageDesc()) ? configMap
				.get(getPackageDesc()) : EMPTY_CONFIG_LIST;
	}

	@Override
	public List<ConfigDesc> getClassConfigDescList(Class<?>... types) {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		addTypeMatchedConfigDesc(list, getClassDesc(), types);
		return list;
	}

	@Override
	public List<ConfigDesc> getConstructorConfigDescList(Class<?>... types) {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		addTypeMatchedConfigDesc(list, getConstructorDesc(), types);
		return list;
	}

	@Override
	public List<ConfigDesc> getMethodConfigDescList(Class<?>... types) {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		for (MethodDesc md : getAllMethodDesc()) {
			addTypeMatchedConfigDesc(list, md, types);
		}
		for (PropertyDesc<T> pd : getAllPropertyDesc()) {
			if (pd.isReadable()) {
				addTypeMatchedConfigDesc(list, pd.getReadMethodDesc(), types);
			}
			if (pd.isWritable()) {
				addTypeMatchedConfigDesc(list, pd.getWriteMethodDesc(), types);
			}
		}
		return list;
	}

	@Override
	public List<ConfigDesc> getPackageConfigDescList(Class<?>... types) {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		addTypeMatchedConfigDesc(list, getPackageDesc(), types);
		return list;
	}

	protected void addTypeMatchedConfigDesc(List<ConfigDesc> list,
			ConfigDescContainer container, Class<?>... types) {
		Assertion.notNull(list);
		Assertion.notNull(container);
		Assertion.notNulls(types);
		for (int i = 0; i < types.length; i++) {
			addTypeMatchedConfigDesc(list, container, types[i]);
		}
	}

	protected void addTypeMatchedConfigDesc(List<ConfigDesc> list,
			ConfigDescContainer container, Class<?> type) {
		for (int i = 0; i < container.getConfigDescSize(); i++) {
			ConfigDesc configDesc = container.getConfigDesc(i);
			if (configDesc.getType().equals(type)) {
				list.add(configDesc);
			}
		}
	}

	@Override
	public Class<? extends T> getConcreteClass() {
		return getClassDesc().getConcreteClass();
	}

	@Override
	public boolean isSingleton() {
		return singleton;
	}

	public void setSingleton(boolean singleton) {
		this.singleton = singleton;
	}
}
