package commons.meta.impl;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Target;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;

import commons.Constants;
import commons.annotation.Lookup;
import commons.bootstrap.CommonsInitializer;
import commons.meta.AnnotationDesc;
import commons.meta.ConfigDescContainer;
import commons.meta.ConfigType;
import commons.meta.LookupContext;
import commons.meta.spi.AnnotationDescSupport;
import commons.util.AnnotationUtil;
import commons.util.ArrayUtil;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.StringUtil;
import commons.util.Reflections.MethodUtil;

/**
 * Concrete class of AnnotationDesc.
 * 
 * @author shot
 * @see commons.meta.AnnotationDesc
 */
public class AnnotationDescImpl implements AnnotationDesc {

	protected Annotation annotation;

	protected ElementType type;

	protected String annotationName;

	protected List<Annotation> metaAnnotationList;

	protected Class<? extends Annotation> annotationType;

	protected ElementType[] acceptableElementTypes;

	protected ConfigDescContainer container;

	protected List<String> methodNameList = CollectionsUtil.newArrayList();

	protected Map<Class<? extends Annotation>, LookupContext> lookupContextMap = CollectionsUtil
			.newHashMap();

	protected static volatile AnnotationDescSupport annotationDescSupport;

	static {
		if (!CommonsInitializer.isInitialized()) {
			CommonsInitializer.init();
		}
		if (annotationDescSupport == null) {
			synchronized (AnnotationDescImpl.class) {
				if (annotationDescSupport == null) {
					annotationDescSupport = CommonsInitializer.getFactory()
							.getAnnotationDescSupport();
				}
			}
		}
	}

	public AnnotationDescImpl(Annotation annotation) {
		this(annotation, null);
	}

	public AnnotationDescImpl(Annotation annotation, ElementType type) {
		this(annotation, type, null);
	}

	public AnnotationDescImpl(Annotation annotation, ElementType type,
			ConfigDescContainer container) {
		this.annotation = Assertion.notNull(annotation);
		this.type = type;
		this.metaAnnotationList = CollectionsUtil.newArrayList();
		this.container = container;
		initAnnotationInfo(annotation);
		initMetaAnnotationList(annotationType);
		initLookupMethodMap(annotationType);
	}

	protected void initAnnotationInfo(Annotation annotation) {
		this.annotationName = annotation.annotationType().getSimpleName();
		this.annotationType = annotation.annotationType();
	}

	protected void initMetaAnnotationList(
			final Class<? extends Annotation> annotationType) {
		for (Annotation a : annotationType.getAnnotations()) {
			if (!AnnotationUtil.isJavaLangAnnotation(a)) {
				metaAnnotationList.add(a);
			}
			if (AnnotationUtil.isTargetAnnotation(a)) {
				Target t = Target.class.cast(a);
				this.acceptableElementTypes = t.value();
				assertAcceptableElementType();
			}
		}
	}

	protected void initLookupMethodMap(
			Class<? extends Annotation> annotationType) {
		for (Method m : annotationType.getDeclaredMethods()) {
			for (Annotation a : m.getDeclaredAnnotations()) {
				if (a.annotationType() == Lookup.class) {
					Lookup lookup = Lookup.class.cast(a);
					final Class<? extends Annotation> key = lookup.value();
					LookupContext context = null;
					if (!lookupContextMap.containsKey(key)) {
						context = new LookupContext(key);
					} else {
						context = lookupContextMap.get(key);
					}
					String alias = lookup.alias();
					if (StringUtil.isEmpty(alias)) {
						alias = m.getName();
					}
					context.addMethodValue(alias, MethodUtil.invoke(m,
							this.annotation, Constants.EMPTY_ARRAY));
					lookupContextMap.put(key, context);
				}
			}
			methodNameList.add(m.getName());
		}
	}

	protected void assertAcceptableElementType() {
		if (acceptableElementTypes == null) {
			return;
		}
		if (type != null && !ArrayUtil.contains(acceptableElementTypes, type)) {
			throw new IllegalStateException(
					"ElementType should be in @Target' element types.");
		}
	}

	@Override
	public ElementType getElementType() {
		return type;
	}

	@Override
	public List<Annotation> getMetaAnnotationList() {
		return metaAnnotationList;
	}

	@Override
	public void setElementType(ElementType type) {
		this.type = type;
		assertAcceptableElementType();
	}

	@Override
	public ElementType[] getAcceptableTypes() {
		return acceptableElementTypes;
	}

	@Override
	public boolean isAcceptableType(ElementType type) {
		Assertion.notNull(type);
		return ArrayUtil.contains(getAcceptableTypes(), type);
	}

	@Override
	public Annotation getAnnotation() {
		return annotation;
	}

	@Override
	public String getName() {
		return this.annotationName;
	}

	@Override
	public Class<? extends Annotation> getType() {
		return annotationType;
	}

	@Override
	public ConfigDescContainer getConfigDescContainer() {
		return container;
	}

	@Override
	public void setConfigDescContainer(ConfigDescContainer container) {
		this.container = Assertion.notNull(container);
	}

	@Override
	public ConfigType getConfigType() {
		return ConfigType.ANNOTATION;
	}

	@Override
	public boolean hasAnnotation() {
		return true;
	}

	/**
	 * Experimental, not stable.
	 */
	@Override
	public void setValue(String methodName, Object value) {
		Assertion.notNull(methodName);
		Assertion.notNull(value);
		synchronized (this.annotation) {
			this.annotation = annotationDescSupport.createAnnotation(
					this.annotation, methodName, value);
			initAnnotationInfo(annotation);
		}
	}

	@Override
	public boolean hasLookup(Class<? extends Annotation> annotationClass) {
		Assertion.notNull(annotationClass);
		return lookupContextMap.containsKey(annotationClass);
	}

	@Override
	public LookupContext getLookupContext(
			Class<? extends Annotation> annotationClass) {
		Assertion.notNull(annotationClass);
		return lookupContextMap.get(annotationClass);
	}

	@Override
	public void setValues(LookupContext lookupContext) {
		Assertion.notNull(lookupContext);
		for (String methodName : methodNameList) {
			if (!lookupContext.contains(methodName)) {
				continue;
			}
			Object value = lookupContext.getValue(methodName);
			setValue(methodName, value);
		}
	}

}
