package commons.meta;

import static commons.util.AnnotationUtil.createAnnotationDescList;
import static java.lang.annotation.ElementType.CONSTRUCTOR;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.PACKAGE;
import static java.lang.annotation.ElementType.TYPE;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import commons.util.CollectionsUtil;

/**
 * Configuration support class.
 * 
 * @author shot
 * 
 */
public class ConfigDescSupport {

	protected final ConfigDescContainer container;

	private static final List<ConfigDesc> EMPTY_LIST = Collections.emptyList();

	protected List<ConfigDesc> annotationDescList = CollectionsUtil
			.newArrayList();

	protected List<ConfigDesc> outerConfigDescList = CollectionsUtil
			.newArrayList();

	protected List<ConfigDesc> configDescList = CollectionsUtil.newArrayList();

	protected Map<Class<?>, List<ConfigDesc>> typeMap = CollectionsUtil
			.newHashMap();

	public ConfigDescSupport(ConfigDescContainer container) {
		this.container = container;
	}

	public void addAllAnnotationDesc(Class<?> targetClass) {
		addAllAnnotationDesc(createAnnotationDescList(container, targetClass,
				TYPE));
	}

	public void addAllAnnotationDesc(Constructor<?>[] constructors) {
		List<ConfigDesc> list = CollectionsUtil.newArrayList();
		for (Constructor<?> c : constructors) {
			list.addAll(createAnnotationDescList(container, c, CONSTRUCTOR));
		}
		addAllAnnotationDesc(list);
	}

	public void addAllAnnotationDesc(Method method) {
		addAllAnnotationDesc(createAnnotationDescList(container, method, METHOD));
	}

	public void addAllAnnotationDesc(Package package_) {
		if (package_ != null) {
			addAllAnnotationDesc(createAnnotationDescList(container, package_,
					PACKAGE));
		}
	}

	public void addAllAnnotationDesc(Field field) {
		addAllAnnotationDesc(createAnnotationDescList(container, field, FIELD));
	}

	protected void addAllAnnotationDesc(List<ConfigDesc> list) {
		for (ConfigDesc configDesc : list) {
			addConfigDesc(configDesc);
		}
	}

	public int getConfigDescSize() {
		return configDescList.size();
	}

	public ConfigDesc getConfigDesc(int index) {
		if (index < getConfigDescSize()) {
			return configDescList.get(index);
		}
		return null;
	}

	public void addConfigDesc(ConfigDesc configDesc) {
		if (isAnnotationType(configDesc)) {
			annotationDescList.add(configDesc);
		} else if (isOuterConfigType(configDesc)) {
			outerConfigDescList.add(configDesc);
		}
		configDescList.add(configDesc);
		List<ConfigDesc> list = typeMap.get(configDesc.getType());
		if (list == null) {
			list = CollectionsUtil.newArrayList();
		}
		list.add(configDesc);
		typeMap.put(configDesc.getType(), list);
	}

	protected boolean isAnnotationType(ConfigDesc configDesc) {
		return configDesc.getConfigType() == ConfigType.ANNOTATION;
	}

	protected boolean isOuterConfigType(ConfigDesc configDesc) {
		return configDesc.getConfigType() == ConfigType.OUTER;
	}

	public List<ConfigDesc> findConfigDescs(Class<?> clazz) {
		return typeMap.containsKey(clazz) ? typeMap.get(clazz) : EMPTY_LIST;
	}

	public boolean hasConfigDesc(Class<?> clazz) {
		return typeMap.containsKey(clazz);
	}

	public List<ConfigDesc> getConfigDescs() {
		return configDescList;
	}

	public ConfigDesc findConfigDesc(Class<?> clazz) {
		List<ConfigDesc> list = findConfigDescs(clazz);
		return (list.size() > 0) ? list.get(0) : null;
	}

}
