package commons.meta;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import commons.util.Assertion;
import commons.util.CollectionsUtil;

/**
 * Support class of configuring BeanDesc.
 * 
 * @author shot
 * 
 * @param <T>
 */
public class BeanDescSupport<T> {

	protected static final Set<Method> ignores = CollectionsUtil.newHashSet();

	static {
		init();
	}

	protected final BeanDesc<T> beanDesc;

	public BeanDescSupport(final BeanDesc<T> beanDesc) {
		this.beanDesc = beanDesc;
	}

	public boolean isIgnorableMethod(final Method m) {
		final String methodName = m.getName();
		if ("isBridge".equals(methodName) || "isSynthetic".equals(methodName)) {
			return true;
		}
		for (Method ignoreMethod : ignores) {
			if (ignoreMethod.equals(m)) {
				return true;
			}
		}
		return false;
	}

	public void addIgnore(Method m) {
		ignores.add(Assertion.notNull(m));
	}

	public void storeAnnotationConfiguration(
			final Map<ConfigDescContainer, List<ConfigDesc>> configMap) {
		addAnnotationDesc(beanDesc.getClassDesc(), configMap);
		addAnnotationDesc(beanDesc.getConstructorDesc(), configMap);
		for (MethodDesc md : beanDesc.getAllMethodDesc()) {
			addAnnotationDesc(md, configMap);
		}
		for (PropertyDesc<T> pd : beanDesc.getAllPropertyDesc()) {
			addAnnotationDesc(pd, configMap);
		}
		if (beanDesc.getPackageDesc() != null) {
			addAnnotationDesc(beanDesc.getPackageDesc(), configMap);
		}
	}

	protected void addAnnotationDesc(ConfigDescContainer container,
			final Map<ConfigDescContainer, List<ConfigDesc>> configMap) {
		List<ConfigDesc> configDescs = container.getConfigDescs();
		List<ConfigDesc> list = configMap.get(container);
		if (list == null) {
			list = CollectionsUtil.newArrayList();
		}
		if (!configDescs.isEmpty()) {
			list.addAll(configDescs);
		}
		configMap.put(container, list);
	}

	public void clear() {
		init();
	}

	private static void init() {
		ignores.addAll(Arrays.asList(Object.class.getDeclaredMethods()));
	}

}
