package commons.dataset.impl.jxl;

import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;

import jxl.CellType;
import jxl.Workbook;
import jxl.WorkbookSettings;
import jxl.read.biff.BiffException;
import jxl.write.WritableWorkbook;
import jxl.write.WriteException;

import commons.dataset.DataType;
import commons.exception.IORuntimeException;
import commons.util.Assertion;
import commons.util.CollectionsUtil;

public class JxlUtil {

	public static final WorkbookSettings settings = new WorkbookSettings();

	public static final Map<CellType, DataType> cellTypeToDataTypeMap = CollectionsUtil
			.newHashMap();

	public static final Map<DataType, CellType> dataTypeToCellTypeMap = CollectionsUtil
			.newHashMap();

	static {
		settings.setGCDisabled(true);
		settings.setLocale(Locale.getDefault());

		cellTypeToDataTypeMap.put(CellType.BOOLEAN, DataType.BOOLEAN);
		cellTypeToDataTypeMap.put(CellType.DATE, DataType.DATE);
		cellTypeToDataTypeMap.put(CellType.LABEL, DataType.STRING);
		cellTypeToDataTypeMap.put(CellType.EMPTY, DataType.STRING);
		cellTypeToDataTypeMap.put(CellType.NUMBER, DataType.NUMBER);

		dataTypeToCellTypeMap.put(DataType.BOOLEAN, CellType.BOOLEAN);
		dataTypeToCellTypeMap.put(DataType.DATE, CellType.DATE);
		dataTypeToCellTypeMap.put(DataType.STRING, CellType.LABEL);
		dataTypeToCellTypeMap.put(DataType.STRING, CellType.EMPTY);
		dataTypeToCellTypeMap.put(DataType.NUMBER, CellType.NUMBER);
	}

	public static Workbook getWorkbook(final File file) {
		Assertion.notNull(file);
		try {
			return Workbook.getWorkbook(file, settings);
		} catch (BiffException e) {
			throw new JXLRuntimeException(e, file);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static WritableWorkbook createWritableWorkbook(final File file) {
		Assertion.notNull(file);
		try {
			return Workbook.createWorkbook(file, settings);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static void write(WritableWorkbook workbook) {
		Assertion.notNull(workbook);
		try {
			workbook.write();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static void close(WritableWorkbook workbook) {
		Assertion.notNull(workbook);
		try {
			workbook.close();
		} catch (WriteException e) {
			// TODO
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	public static DataType getDataType(CellType type) {
		DataType retType = cellTypeToDataTypeMap.get(Assertion.notNull(type));
		return (retType != null) ? retType : DataType.OBJECT;
	}

	public static CellType getCellType(DataType type) {
		CellType retType = dataTypeToCellTypeMap.get(type);
		return (retType != null) ? retType : CellType.LABEL;
	}
}
