package commons.dataset.impl.jxl;

import static commons.util.Assertion.notNull;

import java.io.File;
import java.math.BigDecimal;
import java.util.Date;

import jxl.CellType;
import jxl.write.Label;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import jxl.write.WriteException;

import commons.dataset.DataColumn;
import commons.dataset.DataRow;
import commons.dataset.DataSet;
import commons.dataset.DataTable;
import commons.dataset.DataWriter;
import commons.util.Assertion;
import commons.util.ConverterUtil;

public class ExcelWriter implements DataWriter {

	protected WritableWorkbook workbook;

	public ExcelWriter(final String path) {
		this(new File(notNull(path)));
	}

	public ExcelWriter(final File file) {
		Assertion.notNull(file);
		workbook = JxlUtil.createWritableWorkbook(file);
	}

	@Override
	public void write(DataSet dataSet) {
		Assertion.notNull(dataSet);
		int i = 0;
		for (DataTable table : dataSet.getAllDataTable()) {
			String name = table.getTableName();
			WritableSheet sheet = workbook.createSheet(name, i++);
			for (int rowIndex = 0; rowIndex < table.getRowSize(); ++rowIndex) {
				DataRow row = table.getRow(rowIndex);
				for (int columnIndex = 0; columnIndex < table.getColumnSize(); ++columnIndex) {
					DataColumn column = table.getColumn(columnIndex);
					CellType type = JxlUtil.getCellType(column.getType());
					addCell(sheet, type, row, columnIndex, rowIndex);
				}
			}
		}
		writeToWorkbook();
		closeWorkbook();
	}

	protected void addCell(WritableSheet sheet, CellType type, DataRow row,
			int columnIndex, int rowIndex) {
		try {
			if (type == CellType.BOOLEAN) {
				Boolean b = (Boolean) row.getValue(columnIndex);
				sheet.addCell(new jxl.write.Boolean(columnIndex, rowIndex, b
						.booleanValue()));
			} else if (type == CellType.DATE) {
				Date date = (Date) row.getValue(columnIndex);
				sheet.addCell(new jxl.write.DateTime(columnIndex, rowIndex,
						date));
			} else if (type == CellType.LABEL) {
				String s = (String) row.getValue(columnIndex);
				sheet.addCell(new Label(columnIndex, rowIndex, s));
			} else if (type == CellType.NUMBER) {
				BigDecimal bd = ConverterUtil.convertAsBigDecimal(row
						.getValue(columnIndex));
				// TODO just say fuck to use double not BigDecimal.
				// We will handle it soon.
				sheet.addCell(new jxl.write.Number(columnIndex, rowIndex, bd
						.doubleValue()));
			} else {
				Object o = row.getValue(columnIndex);
				sheet.addCell(new Label(columnIndex, rowIndex, o.toString()));
			}
		} catch (WriteException e) {
			throw new RuntimeException(e);
		}

	}

	protected void writeToWorkbook() {
		JxlUtil.write(workbook);
	}

	protected void closeWorkbook() {
		JxlUtil.close(workbook);
	}
}
