package commons.dataset.impl.jxl;

import java.io.File;

import jxl.Cell;
import jxl.CellType;
import jxl.Sheet;
import jxl.Workbook;

import commons.dataset.DataColumn;
import commons.dataset.DataConverter;
import commons.dataset.DataReader;
import commons.dataset.DataRow;
import commons.dataset.DataSet;
import commons.dataset.DataTable;
import commons.dataset.impl.DataSetImpl;
import commons.util.Assertion;
import commons.util.ConverterUtil;

public class ExcelReader implements DataReader, DataConverter {

	protected DataSet dataSet;

	public ExcelReader(File file) {
		Assertion.notNull(file);
		Workbook workbook = JxlUtil.getWorkbook(file);
		createDataSet();
		initTables(workbook);
		initSheets(workbook.getSheets());
	}

	protected void initSheets(Sheet[] sheets) {
		for (Sheet s : Assertion.notNull(sheets)) {
			String name = s.getName();
			DataTable table = dataSet.getDataTable(name);
			setupTable(table, s);
		}
	}

	protected void setupTable(final DataTable table, final Sheet s) {
		final int rowsLength = getRowsLength(s);
		for (int rowIndex = 0; rowIndex < rowsLength; ++rowIndex) {
			final DataRow row = getRowAsTableBody(rowIndex, table);
			for (int columnIndex = 0; columnIndex < s.getColumns(); ++columnIndex) {
				final Cell cell = s.getCell(columnIndex, rowIndex);
				final CellType type = cell.getType();
				if (rowIndex == 0) {
					setupHeaderColumn(table, cell);
					continue;
				}
				setupDataColumn(table, type, columnIndex);
				setupDataRow(row, cell, type, columnIndex);
			}
		}
	}

	protected void setupDataRow(DataRow row, Cell cell, CellType type,
			int columnIndex) {
		if (type == CellType.LABEL) {
			row.setValue(columnIndex, cell.getContents());
		} else if (type == CellType.NUMBER) {
			final String str = cell.getContents();
			final Object o = ConverterUtil.convertAsBigDecimal(str);
			row.setValue(columnIndex, o);
		} else if (type == CellType.BOOLEAN) {
			final String str = cell.getContents();
			final Boolean b = ConverterUtil.convertAsBoolean(str);
			row.setValue(columnIndex, b);
		} else if (type == CellType.DATE) {
			final String str = cell.getContents();
			row.setValue(columnIndex, ConverterUtil.convertAsDate(str));
		} else if (type == CellType.EMPTY) {
			row.setValue(columnIndex, cell.getContents());
		} else {
			row.setValue(columnIndex, cell.getContents());
		}
	}

	protected DataColumn setupHeaderColumn(final DataTable table,
			final Cell cell) {
		return table.addColumn(cell.getContents());
	}

	protected DataColumn setupDataColumn(DataTable table, CellType type,
			int columnIndex) {
		DataColumn column = table.getColumn(columnIndex);
		setType(type, column);
		return column;
	}

	protected void setType(CellType type, DataColumn column) {
		column.setType(JxlUtil.getDataType(type));
	}

	protected void initTables(Workbook workbook) {
		for (Sheet s : workbook.getSheets()) {
			String name = s.getName();
			dataSet.addDataTable(name);
		}
	}

	@Override
	public DataSet read() {
		return dataSet;
	}

	@Override
	public <T> T convert(DataTable from, Class<T> to) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> DataTable convert(T from) {
		throw new UnsupportedOperationException();
	}

	protected void createDataSet() {
		this.dataSet = new DataSetImpl();
	}

	protected int getRowsLength(Sheet s) {
		return s.getRows();
	}

	protected DataRow getRowAsTableBody(int rowIndex, DataTable table) {
		return (rowIndex != 0) ? table.addRow() : null;
	}

}
