package commons.dataset.impl.csv;

import static commons.util.Assertion.notNull;

import java.io.File;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.List;

import commons.dataset.DataRow;
import commons.dataset.DataSet;
import commons.dataset.DataTable;
import commons.dataset.DataWriter;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.SystemPropertyUtil;
import commons.util.WriterUtil;

public class CsvWriter implements DataWriter {

	public static final char DEFAULT_SEPARATOR = ',';

	public static final char DEFAULT_QUOTE = '"';

	public static final char DEFAULT_ESCAPE = '"';

	public static final char NO_SEPARATOR = '\u0000';

	public static final char NO_ESCAPE = '\u0000';

	protected PrintWriter writer;

	protected boolean noquote = false;

	protected boolean noescape = false;

	public CsvWriter(final String path) {
		this(new File(notNull(path)));
	}

	public CsvWriter(final File file) {
		this(WriterUtil.createPrintWriter(file));
	}

	public CsvWriter(final Writer writer) {
		this(new PrintWriter(writer));
	}

	public CsvWriter(final PrintWriter writer) {
		this.writer = Assertion.notNull(writer);
	}

	@Override
	public void write(DataSet dataSet) {
		Assertion.notNull(dataSet);
		try {
			final DataTable table = dataSet.getDataTable(0);
			for (int i = 0; i < table.getRowSize(); ++i) {
				final DataRow row = table.getRow(i);
				List<String> strs = CollectionsUtil.newArrayList();
				int columnSize = table.getColumnSize();
				for (int columnIndex = 0; columnIndex < columnSize; ++columnIndex) {
					strs.add(row.getValue(columnIndex).toString());
				}
				writeLine(strs);
			}
		} finally {
			flush();
			close();
		}
	}

	protected void writeLine(final List<String> strs) {
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < strs.size(); ++i) {
			if (i != 0) {
				appendSeparator(buf);
			}
			if (!isNoquote()) {
				appendQuote(buf);
			}
			String s = strs.get(i);
			for (int j = 0; j < s.length(); ++j) {
				char c = s.charAt(j);
				if (!isNoescape() && isQuote(c)) {
					appendWithEscape(buf, c);
				} else if (!isNoescape() && isEscape(c)) {
					appendWithEscape(buf, c);
				} else {
					append(buf, c);
				}
			}
			if (!isNoquote()) {
				appendQuote(buf);
			}
		}
		buf.append(SystemPropertyUtil.LINE_SEP);
		writer.write(buf.toString());
	}

	protected void flush() {
		writer.flush();
	}

	protected void close() {
		WriterUtil.close(writer);
		writer = null;
	}

	public PrintWriter getWriter() {
		return writer;
	}

	public boolean isNoquote() {
		return noquote;
	}

	public void setNoquote(boolean noquote) {
		this.noquote = noquote;
	}

	public boolean isNoescape() {
		return noescape;
	}

	public void setNoescape(boolean noescape) {
		this.noescape = noescape;
	}

	protected static void append(final StringBuilder buf, char c) {
		buf.append(c);
	}

	protected static void appendSeparator(final StringBuilder buf) {
		buf.append(DEFAULT_SEPARATOR);
	}

	protected static void appendQuote(final StringBuilder buf) {
		buf.append(DEFAULT_QUOTE);
	}

	protected static void appendLineSeparator(final StringBuilder buf) {
		buf.append(SystemPropertyUtil.LINE_SEP);
	}

	protected static void appendWithEscape(final StringBuilder buf, final char c) {
		buf.append(DEFAULT_ESCAPE).append(c);
	}

	protected static boolean isQuote(char c) {
		return c == DEFAULT_QUOTE;
	}

	protected static boolean isEscape(char c) {
		return c == DEFAULT_ESCAPE;
	}

}
