package commons.dataset.impl.csv;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.List;

import commons.dataset.DataColumn;
import commons.dataset.DataReader;
import commons.dataset.DataRow;
import commons.dataset.DataSet;
import commons.dataset.DataTable;
import commons.dataset.DataType;
import commons.dataset.impl.DataSetImpl;
import commons.exception.IORuntimeException;
import commons.util.Assertion;
import commons.util.CollectionsUtil;
import commons.util.ConverterUtil;
import commons.util.ReaderUtil;
import commons.util.SystemPropertyUtil;

public class CsvReader implements DataReader {

	public static final char DEFAULT_SEPARATOR = ',';

	public static final char DEFAULT_QUOTE = '"';

	public static final String STRING_TABLE_NAME = "str";

	protected DataSet dataSet;

	protected DataType[] types;

	protected BufferedReader reader;

	public CsvReader(final String csv) {
		this(csv, null);
	}

	public CsvReader(final String csv, final DataType[] types) {
		Assertion.notNull(csv);
		this.types = types;
		this.reader = ReaderUtil.createBufferedReader(csv);
		init(STRING_TABLE_NAME, reader);
	}

	public CsvReader(final File file) {
		this(file, null);
	}

	public CsvReader(final File file, final DataType[] types) {
		Assertion.notNull(file);
		this.types = types;
		this.reader = ReaderUtil.createBufferedReader(file);
		init(file.getName(), reader);
	}

	protected void init(final String name, final BufferedReader reader) {
		initDataSet();
		DataTable table = initDataTable(name);
		readAll(reader, table);
	}

	protected void readAll(final BufferedReader reader, final DataTable table) {
		String line = null;
		boolean firstLine = true;
		while ((line = readLine()) != null) {
			String[] strs = parseLine(line);
			Assertion.notNull(strs);
			if (firstLine) {
				setupDataColumn(strs, table);
			}
			setupDataRow(strs, table);
			firstLine = false;
		}
	}

	protected void setupDataColumn(final String[] strs, final DataTable table) {
		if (strs.length <= 0) {
			throw new RuntimeException(
					"the first line shoulbe more than one column.");
		}
		for (int i = 0; i < strs.length; ++i) {
			DataType type = null;
			if (types != null && i < types.length) {
				type = types[i];
			} else {
				type = DataType.STRING;
			}
			table.addColumn(String.valueOf(i), type);
		}
	}

	protected void setupDataRow(final String[] strs, final DataTable table) {
		DataRow row = table.addRow();
		for (int columnIndex = 0; columnIndex < strs.length; ++columnIndex) {
			String s = strs[columnIndex];
			DataColumn column = table.getColumn(columnIndex);
			setValue(row, column.getType(), columnIndex, s);
		}
	}

	protected void setValue(DataRow row, DataType type, int index, String value) {
		if (type == DataType.STRING) {
			row.setValue(index, value);
		} else if (type == DataType.NUMBER) {
			row.setValue(index, ConverterUtil.convertAsBigDecimal(value));
		} else if (type == DataType.BOOLEAN) {
			row.setValue(index, ConverterUtil.convertAsBoolean(value));
		} else if (type == DataType.DATE) {
			row.setValue(index, ConverterUtil.convertAsDate(value));
		} else {
			row.setValue(index, value);
		}
	}

	protected String[] parseLine(String line) {
		List<String> list = CollectionsUtil.newArrayList();
		StringBuilder buf = new StringBuilder();
		boolean quoted = false;
		while (true) {
			if (quoted) {
				buf.append(SystemPropertyUtil.LINE_SEP);
				line = readLine();
				if (line == null) {
					break;
				}
			}
			for (int i = 0, len = line.length(); i < len; ++i) {
				char c = line.charAt(i);
				if (isQuote(c)) {
					boolean hasNextChar = (i + 1) < len;
					if (quoted && hasNextChar && isQuote(line.charAt(i + 1))) {
						buf.append(line.charAt(i + 1));
						++i;
					} else {
						quoted = !quoted;
						if (i > 2 && hasNextChar) {
							char prevchar = line.charAt(i - 1);
							if (!isSeparator(prevchar)
									&& isQuote(line.charAt(i + 1))) {
								buf.append(c);
							}
						}
					}
				} else if (isSeparator(c) && !quoted) {
					list.add(buf.toString());
					buf = new StringBuilder();
				} else {
					buf.append(c);
				}
			}
			// if go to next line, take over StringBuilder.
			if (!quoted) {
				list.add(buf.toString());
				break;
			}
		}
		return list.toArray(new String[list.size()]);
	}

	protected static boolean isQuote(char c) {
		return c == DEFAULT_QUOTE;
	}

	protected static boolean isSeparator(char c) {
		return c == DEFAULT_SEPARATOR;
	}

	protected String readLine() {
		try {
			return reader.readLine();
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	protected DataTable initDataTable(final String name) {
		return dataSet.addDataTable(name);
	}

	protected void initDataSet() {
		dataSet = new DataSetImpl();
	}

	@Override
	public DataSet read() {
		return dataSet;
	}

}
