#include <windows.h>
#include <process.h>
#include <shlobj.h>
#include <stdio.h>
#include <jni.h>

#include "jvm.h"
#include "eventlog.h"
#include "service.h"

BOOL WINAPI HandlerRoutine(DWORD dwCtrlType);
void    InitResource(LPCTSTR name);
DWORD   GetResourceSize(LPCTSTR name);
jbyte*  GetResourceBuffer(LPCTSTR name);
jstring GetJString(const char* value);
LPSTR   GetShiftJIS(jstring src);

void JNICALL JNI_WriteEventLog(JNIEnv *env, jobject clazz, jint logType, jstring message);
void JNICALL JNI_UncaughtException(JNIEnv *env, jobject clazz, jstring message, jstring trace);

TCHAR  cache[] = "12345678901234567890123456789012";
jbyte* buffer = NULL;
DWORD  size   = 0;

static char      service_name[1024];    
static jclass    mainClass;
static jmethodID stopMethod;

int isService = FALSE;
int isRunning = FALSE;

int main(int argc, char* argv[]) {
	if((argc >= 2) && (_stricmp(argv[1], "-service") == 0)) {
		isService = TRUE;
	}
	service_main(argc, argv);
}

int service_install(int argc, char* argv[]) {
	if(GetResourceSize("SVCDESC") > 0) {
		SetServiceDescription((LPTSTR)GetResourceBuffer("SVCDESC"));
	}
	InstallEventLog();
}

int service_remove() {
	RemoveEventLog();
}

int service_start(int argc, char* argv[]) {
	char message[1024];
	LPTSTR vm_args_opt = NULL;
	if(GetResourceSize("VMARGS") > 0) {
		vm_args_opt = (LPTSTR)GetResourceBuffer("VMARGS");
	}
	if(CreateJavaVM(vm_args_opt) == NULL) {
		fputs("JavaVM 쐬ł܂łB\r\n", stderr);
		return -1;
	}
	// Toolkit
	jclass toolkitClass = env->DefineClass("Toolkit", NULL, GetResourceBuffer("TOOLKIT"), GetResourceSize("TOOLKIT"));
	if(toolkitClass == NULL) {
		fputs("Fialed to define class: Toolkit\n", stderr);
		return -2;
	}
	// Loader
	jclass loaderClass = env->DefineClass("Loader", NULL, GetResourceBuffer("LOADER"), GetResourceSize("LOADER"));
	if(loaderClass == NULL) {
		fputs("Fail to define class: Loader\n", stderr);
		return -3;
	}
	jmethodID loaderInit = env->GetMethodID(loaderClass, "<init>", "()V");
	if(loaderInit == NULL) {
		fputs("Fail to find method: Loader#init()\r\n", stderr);
		return -4;
	}
	jobject loader = env->NewObject(loaderClass, loaderInit);
	if(loader == NULL) {
		fputs("Fail to init: Loader\r\n", stderr);
		return -5;
	}
	jmethodID defineClass = env->GetMethodID(loaderClass, "defineClass", "([B)Ljava/lang/Class;");
	if(defineClass == NULL) {
		fputs("Fail to find method: Loader#defineClass(byte[])\n", stderr);
        return -6;
    }
	// Main-Class
	jbyteArray jar = env->NewByteArray(GetResourceSize("JAR"));
	env->SetByteArrayRegion(jar, 0, GetResourceSize("JAR"), GetResourceBuffer("JAR"));

	//eventlog T[rXƂċNĂꍇAo͂CxgOɐ؂ւ܂B
	jclass eventLogStreamClass = NULL;
	if(isService) {
		jmethodID eventLogStreamInit;
		if((eventLogStreamClass = env->DefineClass("EventLogStream", NULL, GetResourceBuffer("EVENTLOG"), GetResourceSize("EVENTLOG"))) != NULL) {
			if((eventLogStreamInit = env->GetStaticMethodID(eventLogStreamClass, "initialize", "()V")) != NULL) {
				env->CallStaticVoidMethod(eventLogStreamClass, eventLogStreamInit);
			}
		}
		
		JNINativeMethod	nm;
		nm.name = "WriteEventLog";
		nm.signature = "(ILjava/lang/String;)V";
		nm.fnPtr = (void*)JNI_WriteEventLog;
		if((env->RegisterNatives(eventLogStreamClass, &nm, 1)) != 0) {
			fputs("Fail to regist native method: WriteEventLog\n", stderr);
			return -6;
		}
	}
	//eventlog
	
	mainClass = (jclass)(env->CallObjectMethod(loader, defineClass, jar));
	if(mainClass == NULL) {
		fputs("Fail to load Main-Class\n", stderr);
		return -8;
	}
	jmethodID startMethod = env->GetStaticMethodID(mainClass, "start", "([Ljava/lang/String;)V");
	if(startMethod == NULL) {
		strcpy(message, "T[rXJnł܂BT[rXvO̓CNX "
						"public static void start(String[] args) Kv܂B");
		if(isService) {
			WriteEventLog(EVENTLOG_ERROR_TYPE, message);
		} else {
			fprintf(stderr, "%s\n", message);
		}
		return -9;
	}
	stopMethod = env->GetStaticMethodID(mainClass, "stop", "()V");
	if(stopMethod == NULL) {
		strcpy(message, "T[rXJnł܂BT[rXvO̓CNX "
						"public static void stop() Kv܂B");
		if(isService) {
			WriteEventLog(EVENTLOG_ERROR_TYPE, message);
		} else {
			fprintf(stderr, "%s\n", message);
		}
		return -9;
	}
	
	// UncaughtHandler (version 1.5.0 or higher only)
	jmethodID loaderGetRuntimeVersion = env->GetStaticMethodID(loaderClass, "getRuntimeVersion", "()I");
	if(loaderGetRuntimeVersion != NULL) {
		jint version = env->CallStaticIntMethod(loaderClass, loaderGetRuntimeVersion);
		if(version >= 150) {
			jclass    uncaughtHandlerClass;
			jmethodID uncaughtHandlerInit;
			if((uncaughtHandlerClass = env->DefineClass("UncaughtHandler", NULL, GetResourceBuffer("HANDLER"), GetResourceSize("HANDLER"))) != NULL) {
				if((uncaughtHandlerInit = env->GetStaticMethodID(uncaughtHandlerClass, "initialize", "()V")) != NULL) {
					env->CallStaticVoidMethod(uncaughtHandlerClass, uncaughtHandlerInit);
					JNINativeMethod	nm;
					nm.name = "UncaughtException";
					nm.signature = "(Ljava/lang/String;Ljava/lang/String;)V";
					nm.fnPtr = (void*)JNI_UncaughtException;
					if((env->RegisterNatives(uncaughtHandlerClass, &nm, 1)) != 0) {
						fputs("Fail to regist native method: UncaughtException\n", stderr);
						return -6;
					}
				}
			}
		}
	}
	//
	
	jobjectArray args;
	if(isService && argc > 2) {
		args = env->NewObjectArray(argc - 2, env->FindClass("java/lang/String"), NULL);
		for(int i = 2; i < argc; i++) {
			env->SetObjectArrayElement(args, (i - 2), GetJString(argv[i]));
		}
	} else if(!isService && argc > 1) {
		args = env->NewObjectArray(argc - 1, env->FindClass("java/lang/String"), NULL);
		for(int i = 1; i < argc; i++) {
			env->SetObjectArrayElement(args, (i - 1), GetJString(argv[i]));
		}
	} else {
		args = env->NewObjectArray(0, env->FindClass("java/lang/String"), NULL);
	}
	
	strcpy(service_name, argv[0]);
	strcpy(message, service_name);
	strcat(message, " T[rX͐ɊJn܂B");
	if(isService) {
		WriteEventLog(EVENTLOG_INFORMATION_TYPE, message);
	} else {
		fprintf(stdout, "%s\n", message);
	}
	
	// JavaVM  CTRL_SHUTDOWN_EVENT 󂯎ďIĂ܂Ȃ悤ɁAnho^Đ肵܂B
	SetConsoleCtrlHandler((PHANDLER_ROUTINE)HandlerRoutine, TRUE);
	// Vbg_EɃ_CAO\Ȃ悤ɂ܂B
	SetProcessShutdownParameters(0x4FF, SHUTDOWN_NORETRY);
	
	env->CallStaticVoidMethod(mainClass, startMethod, args);

	if(env->ExceptionCheck() == JNI_TRUE) {
		if(isService) {
			jthrowable throwable = env->ExceptionOccurred();
			env->ExceptionClear();
			jmethodID getStackTraceMethod = env->GetStaticMethodID(eventLogStreamClass, "getStackTrace", "(Ljava/lang/Throwable;)Ljava/lang/String;");
			jstring s = (jstring)env->CallStaticObjectMethod(eventLogStreamClass, getStackTraceMethod, throwable);
			strcpy(message, service_name);
			strcat(message, " T[rXُ͈I܂B\r\n\r\n");
			strcat(message, GetShiftJIS(s));
			WriteEventLog(EVENTLOG_ERROR_TYPE, message);
			exit(0);
		} else {
			fprintf(stderr, "%s T[rXُ͈I܂B\r\n\r\n", service_name);
			env->ExceptionDescribe();
			env->ExceptionClear();
		}
	} else {
		strcpy(message, service_name);
		strcat(message, " T[rX͐ɒ~܂B");
		if(isService) {
			WriteEventLog(EVENTLOG_INFORMATION_TYPE, message);
		} else {
			fprintf(stdout, "%s\n", message);
		}
	}
	DetachJavaVM();
	//f[ł͂ȂXbh(ƂSwing)cĂƑҋ@ԂɂȂĂ܂߁A
	//T[rXł́ADestroyJavaVM() sȂ悤ɂĂ܂B
	//DestroyJavaVM();
	return 0;
}

int service_stop() {
	JNIEnv* env = AttachJavaVM();
	env->CallStaticVoidMethod(mainClass, stopMethod);
	DetachJavaVM();
	return 0;
}

int service_error() {
	
}

BOOL WINAPI HandlerRoutine(DWORD dwCtrlType) {
	static int ctrl_c = 0;
	
	switch(dwCtrlType) {
		case CTRL_C_EVENT:
			if(ctrl_c++ == 0) { //͏I݂܂B
				printf("CTRL_C: T[rX~ł...\r\n");
				service_stop();
				return TRUE;
			} else {
				printf("CTRL_C: I܂...\r\n");
				return FALSE;
			}
		
		case CTRL_BREAK_EVENT:
			printf("CTRL_BREAK:\r\n");
			return FALSE;

		case CTRL_CLOSE_EVENT:
			printf("CTRL_CLOSE: T[rX~ł...\r\n");
			service_stop();
			return TRUE;

		case CTRL_LOGOFF_EVENT:
			if(isService == FALSE) {
				printf("CTRL_LOGOFF: T[rX~ł...\r\n");
				service_stop();
			}
			return TRUE;

		case CTRL_SHUTDOWN_EVENT:
			if(isService == FALSE) {
				printf("CTRL_SHUTDOWN: T[rX~ł...\r\n");
				service_stop();
			}
			return TRUE;
	}
	return FALSE;
}

void InitResource(LPCTSTR name) {
	HRSRC hrsrc;
	
	if((hrsrc = FindResource(NULL, name, RT_RCDATA)) == NULL) {
		return;
	}
	size = SizeofResource(NULL, hrsrc);
	buffer = (jbyte*)LockResource(LoadResource(NULL, hrsrc));
	strcpy(cache, name);
}

DWORD GetResourceSize(LPCTSTR name) {
	if(strcmp(name, cache) != 0) {
		InitResource(name);
	}
	return size;
}

jbyte* GetResourceBuffer(LPCTSTR name) {
	if(strcmp(name, cache) != 0) {
		InitResource(name);
	}
	return buffer;
}

jstring GetJString(const char* src) {
    if(src == NULL) {
        return NULL;
    }
	int wSize = MultiByteToWideChar(CP_ACP, 0, src, strlen(src), NULL, 0);
	WCHAR wBuf[wSize];
	MultiByteToWideChar(CP_ACP, 0, src,	strlen(src), wBuf, wSize);
	return env->NewString((jchar*)wBuf, wSize);
}

LPSTR GetShiftJIS(jstring src) {
    if(src == NULL) {
        return NULL;
    }
    const jchar* unicode = env->GetStringChars(src, NULL);
    int length = wcslen((wchar_t*)unicode);
    LPSTR ret = (LPSTR)GlobalAlloc(GMEM_FIXED, sizeof(char) * length * 2 + 1);
    ZeroMemory(ret, sizeof(char) * length * 2 + 1);
    WideCharToMultiByte(CP_ACP, 0, (WCHAR*)unicode, length, ret, length * 2 + 1, NULL, NULL);
    env->ReleaseStringChars(src, unicode);
    return ret;
}

void JNICALL JNI_WriteEventLog(JNIEnv *env, jobject clazz, jint logType, jstring message) {
	WORD nType = EVENTLOG_INFORMATION_TYPE;
	switch(logType) {
		case 0: nType = EVENTLOG_INFORMATION_TYPE; break;
		case 1: nType = EVENTLOG_WARNING_TYPE;     break;
		case 2: nType = EVENTLOG_ERROR_TYPE;       break;
	}
	WriteEventLog(nType, GetShiftJIS(message));
}

void JNICALL JNI_UncaughtException(JNIEnv *env, jobject clazz, jstring message, jstring trace) {
	char buffer[2048];
	strcpy(buffer, service_name);
	strcat(buffer, " T[rXُ͈I܂B\r\n\r\n");
	strcat(buffer, GetShiftJIS(trace));
	if(isService) {
		WriteEventLog(EVENTLOG_ERROR_TYPE, buffer);
	} else {
		fprintf(stderr, "%s\r\n", buffer);
	}
	exit(0);
}
